#!/usr/bin/env python3
"""
Initialize the SOTA Tracker database with schema and seed data.

Run this once to set up the database:
    python init_db.py
"""

import json
import sqlite3
from datetime import datetime
from pathlib import Path

PROJECT_DIR = Path(__file__).parent
DATA_DIR = PROJECT_DIR / "data"
DB_PATH = DATA_DIR / "sota.db"

# Ensure data directory exists
DATA_DIR.mkdir(exist_ok=True)


def create_schema(db: sqlite3.Connection):
    """Create database schema."""
    db.executescript("""
        -- Models table (updated with is_open_source)
        CREATE TABLE IF NOT EXISTS models (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            category TEXT NOT NULL,
            release_date DATE,
            source TEXT DEFAULT 'manual',
            source_url TEXT,
            is_sota BOOLEAN DEFAULT FALSE,
            is_open_source BOOLEAN DEFAULT TRUE,
            sota_rank INTEGER,
            sota_rank_open INTEGER,      -- Rank among open-source only
            metrics JSON,
            last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );

        -- Forbidden models
        CREATE TABLE IF NOT EXISTS forbidden (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            reason TEXT NOT NULL,
            replacement TEXT,
            category TEXT,
            deprecated_date DATE,
            added_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );

        -- Categories
        CREATE TABLE IF NOT EXISTS categories (
            id TEXT PRIMARY KEY,
            name TEXT NOT NULL,
            description TEXT,
            default_open_source BOOLEAN DEFAULT TRUE
        );

        -- Cache tracking (per-category freshness)
        CREATE TABLE IF NOT EXISTS cache_status (
            category TEXT PRIMARY KEY,
            last_fetched TIMESTAMP,
            fetch_source TEXT,           -- 'huggingface', 'lmarena', 'artificial_analysis', 'manual'
            fetch_success BOOLEAN DEFAULT TRUE,
            error_message TEXT
        );

        -- Indexes
        CREATE INDEX IF NOT EXISTS idx_models_category ON models(category);
        CREATE INDEX IF NOT EXISTS idx_models_sota ON models(is_sota);
        CREATE INDEX IF NOT EXISTS idx_models_open ON models(is_open_source);
        CREATE INDEX IF NOT EXISTS idx_models_name ON models(name);
    """)


def seed_categories(db: sqlite3.Connection):
    """Seed model categories."""
    categories = [
        ("image_gen", "Image Generation", "Text/image to image generation", True),
        ("image_edit", "Image Editing", "Image editing, inpainting, style transfer", True),
        ("video", "Video Generation", "Text/image to video", True),
        ("llm_local", "Local LLMs", "LLMs for local inference (GGUF, llama.cpp)", True),
        ("llm_api", "API LLMs", "Cloud-based LLM APIs", False),
        ("llm_coding", "Coding LLMs", "LLMs optimized for code generation", True),
        ("tts", "Text-to-Speech", "Voice synthesis and cloning", True),
        ("stt", "Speech-to-Text", "Audio transcription", True),
        ("music", "Music Generation", "AI music creation", True),
        ("video2audio", "Video-to-Audio", "Generate synchronized audio from video (foley, SFX)", True),
        ("3d", "3D Generation", "3D model and scene generation", True),
        ("embeddings", "Embeddings", "Vector embedding models", True),
    ]

    db.executemany(
        "INSERT OR REPLACE INTO categories (id, name, description, default_open_source) VALUES (?, ?, ?, ?)",
        categories
    )


def seed_sota_models(db: sqlite3.Connection):
    """Seed current SOTA models (January 2026) - VERIFIED."""

    models = [
        # =====================================================================
        # IMAGE GENERATION (verified Jan 2026)
        # =====================================================================
        # Closed source
        {
            "id": "gpt-image-1.5",
            "name": "GPT Image 1.5",
            "category": "image_gen",
            "release_date": "2025-12-16",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 1,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 overall (1264 Elo), 4x faster than GPT-4o image, native text rendering",
                "why_sota": "Highest benchmark scores, best prompt adherence, superior text-in-image",
                "strengths": ["Text rendering", "Prompt accuracy", "Speed", "Photorealism"],
                "use_cases": ["Marketing assets", "UI mockups", "Any task needing text in images"],
                "elo": 1264
            }
        },
        {
            "id": "gemini-3-pro-image",
            "name": "Gemini 3 Pro Image",
            "category": "image_gen",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 2,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 overall (1235 Elo), strong across all categories, great diversity",
                "why_sota": "Excellent balance of quality, diversity, and consistency",
                "strengths": ["Diverse styles", "Consistency", "Creative interpretation"],
                "use_cases": ["Creative projects", "Varied style needs", "Exploration"],
                "elo": 1235
            }
        },
        # Open source
        {
            "id": "z-image-turbo",
            "name": "Z-Image-Turbo",
            "category": "image_gen",
            "release_date": "2025-11-26",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source (1150 Elo), 6B params, runs on 16GB VRAM, ~$5/1k images self-hosted",
                "why_sota": "Best quality/efficiency ratio for open-source, production-ready speed",
                "strengths": ["Fast inference", "Low VRAM", "Cost-effective", "RTX 4090 compatible"],
                "use_cases": ["High-volume production", "Real-time apps", "Budget-conscious deployments"],
                "vram": "16GB",
                "elo": 1150
            }
        },
        {
            "id": "flux2-dev",
            "name": "FLUX.2-dev",
            "category": "image_gen",
            "release_date": "2025-11-25",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source, 32B params, multi-image consistency, needs 96GB VRAM (A100/H100)",
                "why_sota": "Best for consistent character/style across multiple images",
                "strengths": ["Character consistency", "Style consistency", "High detail", "Professional quality"],
                "use_cases": ["Comic/story generation", "Brand assets", "Character design", "Multi-shot campaigns"],
                "vram": "96GB"
            }
        },
        {
            "id": "qwen-image-2512",
            "name": "Qwen Image (Alibaba, Dec 2025)",
            "category": "image_gen",
            "release_date": "2025-12-31",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source, best human realism/skin detail, 40GB+ VRAM, slow but highest quality portraits",
                "why_sota": "Unmatched photorealistic humans, skin texture, and facial detail",
                "strengths": ["Portrait quality", "Skin realism", "Human anatomy", "Natural lighting"],
                "use_cases": ["Portrait photography", "Fashion", "Character art", "Realistic humans"],
                "vram": "40GB+",
                "constraints": {
                    "cfg": {"min": 3.0, "max": 5.0, "default": 3.5},
                    "steps": {"min": 35, "max": 60, "default": 50},
                    "sampler": "euler",
                    "scheduler": "simple",
                    "shift": {"min": 7.0, "max": 13.0, "default": 7.0, "note": "CRITICAL: shift=3.1 causes blurry output. Use 7.0 for sharp, 12-13 for max sharpness"},
                    "resolution": {"native": [1296, 1296], "divisible_by": 8}
                }
            }
        },

        # =====================================================================
        # IMAGE EDITING
        # =====================================================================
        {
            "id": "qwen-image-edit-2511",
            "name": "Qwen Image Edit (Alibaba, Nov 2025)",
            "category": "image_edit",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 image editing, natural language instructions, preserves unedited regions perfectly",
                "why_sota": "Best understanding of natural language edit requests, surgical precision",
                "strengths": ["Instruction following", "Region preservation", "Natural edits", "Multi-step editing"],
                "use_cases": ["Photo retouching", "Object removal/addition", "Background changes", "Style transfer"]
            }
        },
        {
            "id": "flux1-kontext",
            "name": "FLUX.1-Kontext (BFL)",
            "category": "image_edit",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 image editing, specialized in color/style transfer, maintains structure",
                "why_sota": "Best for recoloring and style changes while preserving composition",
                "strengths": ["Color transfer", "Style transfer", "Structure preservation", "Lighting changes"],
                "use_cases": ["Recoloring products", "Season changes", "Time-of-day changes", "Style variations"]
            }
        },
        {
            "id": "instructpix2pix",
            "name": "InstructPix2Pix",
            "category": "image_edit",
            "release_date": "2023-08-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 image editing, instruction-based editing, fast and lightweight",
                "why_sota": "Pioneer of instruction-based editing, fast inference, well-supported",
                "strengths": ["Fast", "Lightweight", "Well-documented", "Easy to use"],
                "use_cases": ["Quick edits", "Style changes", "Simple modifications", "Prototyping"]
            }
        },
        {
            "id": "adobe-firefly-edit",
            "name": "Adobe Firefly (Edit)",
            "category": "image_edit",
            "release_date": "2025-03-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 4,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 commercial image editing, Photoshop integration, generative fill",
                "why_sota": "Best professional integration, Photoshop native, commercially safe",
                "strengths": ["Photoshop integration", "Commercially safe", "Generative fill", "Professional"],
                "use_cases": ["Professional editing", "Photoshop workflows", "Commercial projects", "Enterprise"]
            }
        },
        {
            "id": "ideogram-edit",
            "name": "Ideogram Edit",
            "category": "image_edit",
            "release_date": "2025-01-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 5,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 commercial image editing, best text preservation during edits",
                "why_sota": "Superior at preserving text when editing images with typography",
                "strengths": ["Text preservation", "Typography", "Web interface", "Quality"],
                "use_cases": ["Marketing materials", "Posters", "Typography edits", "Branding"]
            }
        },

        # =====================================================================
        # VIDEO GENERATION (verified Jan 2026)
        # =====================================================================
        # Closed source
        {
            "id": "veo-3.1",
            "name": "Google Veo 3.1",
            "category": "video",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 1,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 overall, 4K photorealism, native audio, trained on YouTube dataset",
                "why_sota": "Unmatched photorealism from massive YouTube training data, native audio generation",
                "strengths": ["Photorealism", "Physics accuracy", "Native audio", "Long coherence"],
                "use_cases": ["Professional video production", "Commercials", "Film VFX", "High-end content"]
            }
        },
        {
            "id": "runway-gen-4.5",
            "name": "Runway Gen-4.5",
            "category": "video",
            "release_date": "2025-12-11",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 2,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 overall, native audio, 1-min multi-shot videos, character consistency across shots",
                "why_sota": "Best for professional workflows with multi-shot coherence and character tracking",
                "strengths": ["Multi-shot consistency", "Character tracking", "Native audio", "Professional UI"],
                "use_cases": ["Short films", "Music videos", "Ads with recurring characters", "Storyboarding"]
            }
        },
        {
            "id": "sora-2",
            "name": "OpenAI Sora 2",
            "category": "video",
            "release_date": "2025-11-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 3,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#3 overall, native audio, 20s high-quality videos, cinematic camera control",
                "why_sota": "Best understanding of cinematic language, camera movements, and scene composition",
                "strengths": ["Cinematic quality", "Camera control", "Scene composition", "Native audio"],
                "use_cases": ["Cinematic shorts", "Concept visualization", "Creative exploration"]
            }
        },
        {
            "id": "kling-2.6",
            "name": "Kling 2.6",
            "category": "video",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 4,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#4 overall, best human motion/expressions, 1080p, up to 3-min with extend feature",
                "why_sota": "Most realistic human movements, facial expressions, and lip sync",
                "strengths": ["Human motion", "Facial expressions", "Lip sync", "Length extension"],
                "use_cases": ["Talking head videos", "Human-centric content", "Virtual avatars", "Long-form content"]
            }
        },
        # Open source
        {
            "id": "ltx-2",
            "name": "LTX-2",
            "category": "video",
            "release_date": "2026-01-06",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source, native audio, 4K 50fps, runs on 12GB VRAM (RTX 4090), fastest inference",
                "why_sota": "Best open-source with native audio, runs on consumer hardware, real-time capable",
                "strengths": ["Speed", "Native audio", "4K output", "Low VRAM", "RTX 4090 compatible"],
                "use_cases": ["Real-time generation", "High-volume production", "Consumer GPU workflows", "Prototyping"],
                "vram": "12GB"
            }
        },
        {
            "id": "wan-2.2",
            "name": "Wan 2.2 (Alibaba)",
            "category": "video",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 6,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source video (Alibaba Tongyi Wanxiang), MoE architecture, best human motion",
                "why_sota": "Highest quality human motion in open-source, sophisticated MoE architecture",
                "strengths": ["Human motion quality", "Natural movement", "Gesture accuracy", "Emotional expression"],
                "use_cases": ["Human-centric videos", "Dance/movement", "Character animation", "Realistic humans"]
            }
        },
        {
            "id": "hunyuan-video-1.5",
            "name": "HunyuanVideo 1.5",
            "category": "video",
            "release_date": "2025-11-21",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 7,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source, 13B params, cinematic quality, excels at complex multi-object scenes",
                "why_sota": "Best at complex scenes with multiple interacting objects and characters",
                "strengths": ["Complex scenes", "Multi-object tracking", "Cinematic look", "Scene coherence"],
                "use_cases": ["Complex narratives", "Multi-character scenes", "Action sequences", "Detailed environments"]
            }
        },
        {
            "id": "wan-2.1-i2v",
            "name": "Wan 2.1 I2V (Alibaba)",
            "category": "video",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 8,
            "sota_rank_open": 4,
            "metrics": {
                "notes": "#4 open-source video (Alibaba), image-to-video specialist, best creature/animal realism",
                "why_sota": "Unmatched creature animation and texture detail from static images",
                "strengths": ["Image-to-video", "Creature realism", "Texture detail", "Smooth motion"],
                "use_cases": ["Animating photos", "Creature/animal content", "Nature scenes", "Texture-heavy subjects"]
            }
        },

        # =====================================================================
        # LOCAL LLMs (verified Jan 2026)
        # is_uncensored: true for abliterated/JOSIEFIED variants, false for official
        # uncensored_variant: HuggingFace repo for recommended uncensored version
        # vram_gb: numeric VRAM for hardware-aware filtering
        # =====================================================================

        # --- OFFICIAL MODELS (is_uncensored: false) ---
        {
            "id": "qwen3-32b",
            "name": "Qwen 3-32B",
            "category": "llm_local",
            "release_date": "2026-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 local LLM, best single-GPU model, beats GPT-4o on most benchmarks",
                "why_sota": "New benchmark leader, fits RTX 4090/5090 with excellent quality",
                "strengths": ["Benchmark scores", "Hybrid reasoning", "Single GPU", "Multilingual"],
                "use_cases": ["Daily assistant", "Complex reasoning", "RTX 4090/5090 optimal"],
                "vram_gb": 19,
                "quantization": "Q4_K_M",
                "is_uncensored": False,
                "uncensored_variant": "mradermacher/Qwen3-32B-Uncensored-GGUF"
            }
        },
        {
            "id": "qwen3-30b-a3b",
            "name": "Qwen 3-30B-A3B (MoE)",
            "category": "llm_local",
            "release_date": "2026-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 local LLM, 30B MoE with only 3B active, extremely fast inference",
                "why_sota": "Best speed/quality ratio, MoE efficiency lets it run alongside other GPU tasks",
                "strengths": ["Speed", "MoE efficiency", "Low active VRAM", "Multi-tasking friendly"],
                "use_cases": ["Interactive chat", "Running alongside image gen", "Real-time applications"],
                "vram_gb": 18,
                "quantization": "Q4_K_M",
                "active_params": "3B",
                "is_uncensored": False
            }
        },
        {
            "id": "qwen3-8b",
            "name": "Qwen 3-8B",
            "category": "llm_local",
            "release_date": "2026-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 local LLM, best small model, rivals larger 32B models from 2024",
                "why_sota": "Incredible quality at 8B size, fits alongside image/video gen workloads",
                "strengths": ["Small footprint", "Quality/size ratio", "Concurrent workloads", "Fast"],
                "use_cases": ["Daily chat while running FLUX/Wan", "Laptops", "Edge devices"],
                "vram_gb": 6,
                "quantization": "Q5_K_M",
                "is_uncensored": False,
                "uncensored_variant": "mradermacher/Josiefied-Qwen3-8B-abliterated-v1-GGUF"
            }
        },
        {
            "id": "deepseek-r1-32b",
            "name": "DeepSeek R1-32B (Reasoning)",
            "category": "llm_local",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 4,
            "metrics": {
                "notes": "#4 local LLM, specialized reasoning model, o1-level on math/logic",
                "why_sota": "Best open-source reasoning model, excels at multi-step problems",
                "strengths": ["Math reasoning", "Logic", "Chain-of-thought", "Problem solving"],
                "use_cases": ["Complex reasoning", "Math problems", "Code debugging", "Analysis"],
                "vram_gb": 22,
                "quantization": "Q5_K_M",
                "is_uncensored": False
            }
        },
        {
            "id": "qwq-32b",
            "name": "QwQ-32B (Reasoning)",
            "category": "llm_local",
            "release_date": "2025-11-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 5,
            "metrics": {
                "notes": "#5 local LLM, Alibaba's reasoning model, competitive with DeepSeek-R1",
                "why_sota": "Strong reasoning with slightly better general knowledge than R1",
                "strengths": ["Reasoning", "General knowledge", "Instruction following"],
                "use_cases": ["Complex reasoning", "Research", "Analysis"],
                "vram_gb": 19,
                "quantization": "Q4_K_M",
                "is_uncensored": False,
                "uncensored_variant": "mradermacher/QwQ-32B-Uncensored-GGUF"
            }
        },
        {
            "id": "llama3.3-70b",
            "name": "Llama 3.3-70B (Meta)",
            "category": "llm_local",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 6,
            "sota_rank_open": 6,
            "metrics": {
                "notes": "#6 local LLM, GPT-4 class, needs 40GB+ VRAM or dual GPU setup",
                "why_sota": "Meta's most capable open model, enterprise-grade safety",
                "strengths": ["GPT-4 class", "Safety", "Instruction following"],
                "use_cases": ["Enterprise", "Safety-critical apps", "64GB MacBook Pro"],
                "vram_gb": 40,
                "quantization": "Q4_K_M",
                "is_uncensored": False
            }
        },
        {
            "id": "deepseek-v3",
            "name": "DeepSeek V3 (MoE)",
            "category": "llm_local",
            "release_date": "2025-12-25",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 7,
            "sota_rank_open": 7,
            "metrics": {
                "notes": "#7 local LLM, 671B MoE (37B active), needs multi-GPU or API access",
                "why_sota": "Most capable open model overall, but requires significant hardware",
                "strengths": ["Raw capability", "Code", "Math", "Multi-GPU clusters"],
                "use_cases": ["API fallback", "Multi-GPU setups", "Maximum quality needs"],
                "vram_gb": 37,
                "quantization": "Q4_K_M",
                "is_uncensored": False,
                "note": "Requires multi-GPU for local deployment"
            }
        },

        # --- UNCENSORED VARIANTS (is_uncensored: true) ---
        # These are abliterated/JOSIEFIED versions of official models
        {
            "id": "josiefied-qwen3-8b",
            "name": "JOSIEFIED-Qwen3-8B",
            "category": "llm_local",
            "release_date": "2026-01-05",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 8,
            "sota_rank_open": 8,
            "metrics": {
                "notes": "Uncensored Qwen3-8B, JOSIEFIED abliteration + fine-tuning for better quality",
                "why_sota": "Best uncensored small model, JOSIEFIED preserves quality better than pure abliteration",
                "strengths": ["Uncensored", "Small footprint", "Quality preserved", "Creative freedom"],
                "use_cases": ["Creative writing", "Roleplay", "Unrestricted chat", "Running alongside GPU tasks"],
                "vram_gb": 6,
                "quantization": "Q5_K_M",
                "is_uncensored": True,
                "hf_repo": "mradermacher/Josiefied-Qwen3-8B-abliterated-v1-GGUF",
                "base_model": "qwen3-8b"
            }
        },
        {
            "id": "qwen3-32b-uncensored",
            "name": "Qwen3-32B-Uncensored",
            "category": "llm_local",
            "release_date": "2026-01-05",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 9,
            "sota_rank_open": 9,
            "metrics": {
                "notes": "Uncensored Qwen3-32B, abliterated variant without safety guardrails",
                "why_sota": "Full Qwen3-32B quality without restrictions",
                "strengths": ["Uncensored", "Full 32B quality", "Creative freedom", "Single GPU"],
                "use_cases": ["Creative writing", "Unrestricted research", "Roleplay"],
                "vram_gb": 19,
                "quantization": "Q4_K_M",
                "is_uncensored": True,
                "hf_repo": "mradermacher/Qwen3-32B-Uncensored-GGUF",
                "base_model": "qwen3-32b"
            }
        },
        {
            "id": "qwq-32b-uncensored",
            "name": "QwQ-32B-Uncensored",
            "category": "llm_local",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 10,
            "sota_rank_open": 10,
            "metrics": {
                "notes": "Uncensored QwQ-32B reasoning model, abliterated variant",
                "why_sota": "Best uncensored reasoning model, full QwQ quality without restrictions",
                "strengths": ["Uncensored", "Reasoning", "No refusals on complex topics"],
                "use_cases": ["Unrestricted reasoning", "Research", "Complex analysis without guardrails"],
                "vram_gb": 19,
                "quantization": "Q4_K_M",
                "is_uncensored": True,
                "hf_repo": "mradermacher/QwQ-32B-Uncensored-GGUF",
                "base_model": "qwq-32b"
            }
        },

        # =====================================================================
        # API LLMs (closed source)
        # =====================================================================
        {
            "id": "claude-opus-4.5",
            "name": "Claude Opus 4.5",
            "category": "llm_api",
            "release_date": "2025-11-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 1,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 API LLM for complex reasoning, agentic tasks, and nuanced writing (Anthropic)",
                "why_sota": "Best at multi-step reasoning, code understanding, and maintaining context in long conversations",
                "strengths": ["Complex reasoning", "Code generation", "Long context retention", "Agentic workflows"],
                "use_cases": ["Software engineering", "Research assistance", "Complex analysis", "Agentic coding"]
            }
        },
        {
            "id": "gpt-4.5",
            "name": "GPT-4.5",
            "category": "llm_api",
            "release_date": "2025-10-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 2,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 API LLM, strongest multimodal (vision, audio, image gen), OpenAI ecosystem",
                "why_sota": "Best integrated multimodal experience with vision, audio, and generation in one model",
                "strengths": ["Multimodal", "Vision", "Audio", "Image generation", "Ecosystem"],
                "use_cases": ["Multimodal apps", "Vision analysis", "Content creation", "ChatGPT plugins"]
            }
        },
        {
            "id": "gemini-2.0-pro",
            "name": "Gemini 2.0 Pro",
            "category": "llm_api",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 3,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#3 API LLM, 1M+ token context, best for document analysis and research (Google)",
                "why_sota": "Largest context window, best for processing entire codebases or document collections",
                "strengths": ["1M+ context", "Document analysis", "Research", "Google integration"],
                "use_cases": ["Large document analysis", "Codebase review", "Research synthesis", "Book analysis"]
            }
        },
        {
            "id": "grok-4.1",
            "name": "Grok 4.1 (xAI)",
            "category": "llm_api",
            "release_date": "2025-11-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 4,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#4 API LLM, #1 LMArena Elo, real-time X/Twitter data, uncensored",
                "why_sota": "Highest human preference scores, real-time information access, fewer content restrictions",
                "strengths": ["Human preference", "Real-time data", "Uncensored", "X integration"],
                "use_cases": ["Current events", "Social analysis", "Less filtered responses", "Real-time research"]
            }
        },
        {
            "id": "deepseek-v3-api",
            "name": "DeepSeek V3 API (DeepSeek)",
            "category": "llm_api",
            "release_date": "2025-12-25",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source API, $0.07/M input tokens, open weights available, GPT-4 class",
                "why_sota": "Best cost/performance ratio, open weights for self-hosting option",
                "strengths": ["Cheapest API", "Open weights", "GPT-4 class", "Self-host option"],
                "use_cases": ["Cost-sensitive apps", "High-volume generation", "Self-hosting backup", "Budget projects"]
            }
        },
        {
            "id": "mistral-large-api",
            "name": "Mistral Large API (Mistral)",
            "category": "llm_api",
            "release_date": "2025-09-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 6,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source API (EU), 123B MoE, strong multilingual, GDPR compliant",
                "why_sota": "Best European option with GDPR compliance, strong multilingual support",
                "strengths": ["EU-based", "GDPR compliant", "Multilingual", "MoE efficient"],
                "use_cases": ["European deployments", "GDPR requirements", "Multilingual apps", "Enterprise EU"]
            }
        },
        {
            "id": "llama-3.3-together",
            "name": "Llama 3.3-70B via Together.ai",
            "category": "llm_api",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 7,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source API, Meta's Llama hosted by Together, $0.88/M tokens",
                "why_sota": "Best hosted Llama option with competitive pricing and reliability",
                "strengths": ["Llama quality", "Together reliability", "Open weights", "Good pricing"],
                "use_cases": ["Llama fans", "Open-source preference", "Self-host fallback", "Research"]
            }
        },

        # =====================================================================
        # CODING LLMs (NEW CATEGORY)
        # =====================================================================
        {
            "id": "qwen3-coder",
            "name": "Qwen 3 Coder (Alibaba)",
            "category": "llm_coding",
            "release_date": "2026-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source coder, 262K native context (1M with YaRN), beats GPT-4 on HumanEval",
                "why_sota": "Largest code context window in open-source, excellent at multi-file understanding",
                "strengths": ["262K context", "Multi-file projects", "Codebase understanding", "HumanEval leader"],
                "use_cases": ["Large codebase work", "Refactoring", "Code review", "IDE integration"]
            }
        },
        {
            "id": "deepseek-v3-coder",
            "name": "DeepSeek V3 (DeepSeek)",
            "category": "llm_coding",
            "release_date": "2025-12-25",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source coder, 78% MMLU-Pro CS, cheapest capable coding model at $0.07/M tokens",
                "why_sota": "Best cost/performance for code generation, strong at algorithmic problems",
                "strengths": ["Cost efficiency", "Algorithms", "Math/CS", "API availability"],
                "use_cases": ["Budget coding assistant", "LeetCode-style problems", "Algorithm design", "API coding"]
            }
        },
        {
            "id": "claude-opus-4.5-coding",
            "name": "Claude Opus 4.5",
            "category": "llm_coding",
            "release_date": "2025-11-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 3,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 closed-source coder, best at complex multi-step code reasoning and agentic coding",
                "why_sota": "Unmatched at understanding complex codebases and making coordinated multi-file changes",
                "strengths": ["Complex reasoning", "Multi-file edits", "Agentic coding", "Code explanation"],
                "use_cases": ["Claude Code CLI", "Complex refactoring", "Architecture decisions", "Code review"]
            }
        },
        {
            "id": "cursor-claude",
            "name": "Cursor (Claude-based)",
            "category": "llm_coding",
            "release_date": "2025-12-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 4,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 closed-source coder, IDE-integrated with Claude, best autocomplete and inline editing",
                "why_sota": "Best IDE integration with context-aware autocomplete and inline code editing",
                "strengths": ["IDE integration", "Autocomplete", "Inline editing", "Context awareness"],
                "use_cases": ["VS Code replacement", "Real-time coding assistance", "Autocomplete", "Quick edits"]
            }
        },
        {
            "id": "starcoder2-15b",
            "name": "StarCoder2-15B (BigCode)",
            "category": "llm_coding",
            "release_date": "2024-02-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source coder, 16K context, 600+ languages, trained on The Stack v2",
                "why_sota": "Widest language support (600+), best for polyglot codebases",
                "strengths": ["600+ languages", "Fast inference", "Low VRAM", "Well-documented"],
                "use_cases": ["Multi-language projects", "Code completion", "Smaller deployments", "Edge devices"]
            }
        },

        # =====================================================================
        # TTS
        # =====================================================================
        {
            "id": "chatterbox-tts",
            "name": "ChatterboxTTS",
            "category": "tts",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source TTS, most natural prosody and emotion, runs locally",
                "why_sota": "Best naturalness in open-source, handles emotion and emphasis well",
                "strengths": ["Natural prosody", "Emotion", "Local deployment", "Fast inference"],
                "use_cases": ["Audiobooks", "Voice assistants", "Content creation", "Accessibility"]
            }
        },
        {
            "id": "f5-tts",
            "name": "F5-TTS",
            "category": "tts",
            "release_date": "2025-08-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source TTS, best voice cloning from short samples (3-10s reference audio)",
                "why_sota": "Highest quality voice cloning with minimal reference audio needed",
                "strengths": ["Voice cloning", "Short reference audio", "Quality", "Multilingual"],
                "use_cases": ["Voice cloning", "Personalized TTS", "Character voices", "Dubbing"]
            }
        },
        {
            "id": "elevenlabs",
            "name": "ElevenLabs",
            "category": "tts",
            "release_date": "2025-01-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 3,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 commercial TTS, best overall quality, instant voice cloning, many languages",
                "why_sota": "Highest quality output, best voice cloning, widest language support",
                "strengths": ["Quality", "Voice cloning", "Languages", "API ease"],
                "use_cases": ["Professional production", "Multilingual content", "Voice cloning", "Enterprise"]
            }
        },
        {
            "id": "bark",
            "name": "Bark (Suno)",
            "category": "tts",
            "release_date": "2024-04-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source TTS, generates speech + music + sound effects, highly expressive",
                "why_sota": "Most expressive open-source TTS, can generate non-speech audio",
                "strengths": ["Expressiveness", "Sound effects", "Music capability", "Creative"],
                "use_cases": ["Creative projects", "Sound design", "Expressive narration", "Games"]
            }
        },
        {
            "id": "coqui-xtts-v2",
            "name": "Coqui XTTS v2",
            "category": "tts",
            "release_date": "2024-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 4,
            "metrics": {
                "notes": "#4 open-source TTS, 17 languages, cross-lingual voice cloning",
                "why_sota": "Best multilingual open-source with cross-lingual voice cloning",
                "strengths": ["17 languages", "Cross-lingual cloning", "Open-source", "Good quality"],
                "use_cases": ["Multilingual projects", "Voice cloning", "Localization", "International content"]
            }
        },
        {
            "id": "qwen3-tts",
            "name": "Qwen3-TTS (Alibaba)",
            "category": "tts",
            "release_date": "2026-01-15",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "download_url": "https://github.com/QwenLM/Qwen3-TTS",
            "metrics": {
                "notes": "#3 open-source TTS, 10 languages, voice design from text description, 3s rapid clone",
                "why_sota": "Newest SOTA with voice design capability - create voices from text descriptions",
                "strengths": ["Voice design", "10 languages", "3s voice cloning", "Cross-lingual", "Fine-tuning support"],
                "use_cases": ["Custom voice creation", "Character voices", "Multilingual content", "Voice cloning"],
                "vram": "4-6GB"
            }
        },

        # =====================================================================
        # STT (Speech-to-Text)
        # =====================================================================
        {
            "id": "whisper-large-v3",
            "name": "Whisper Large v3 (OpenAI)",
            "category": "stt",
            "release_date": "2024-11-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 STT overall, best accuracy across 99 languages, open-weights",
                "why_sota": "Lowest word error rate, handles accents/noise well, industry standard",
                "strengths": ["Accuracy", "99 languages", "Noise robustness", "Local deployment"],
                "use_cases": ["Transcription", "Meeting notes", "Subtitle generation", "Voice commands"]
            }
        },
        {
            "id": "canary-1b",
            "name": "Canary 1B (NVIDIA)",
            "category": "stt",
            "release_date": "2025-03-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 STT, optimized for NVIDIA GPUs, fast inference with TensorRT",
                "why_sota": "Best performance on NVIDIA hardware, optimized for production",
                "strengths": ["NVIDIA optimized", "Fast inference", "TensorRT", "Production-ready"],
                "use_cases": ["Real-time transcription", "NVIDIA deployments", "Low-latency apps"]
            }
        },
        {
            "id": "faster-whisper-large-v3",
            "name": "Faster Whisper Large v3",
            "category": "stt",
            "release_date": "2024-12-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 STT, CTranslate2 optimized Whisper, 4x faster than original",
                "why_sota": "Same accuracy as Whisper but 4x faster, lower memory usage",
                "strengths": ["4x faster", "Lower memory", "Same accuracy", "CPU efficient"],
                "use_cases": ["Batch processing", "CPU deployment", "Resource-constrained environments"]
            }
        },
        {
            "id": "deepgram-nova-3",
            "name": "Deepgram Nova 3",
            "category": "stt",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 4,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 commercial STT API, real-time streaming, speaker diarization",
                "why_sota": "Best commercial API with real-time streaming and advanced features",
                "strengths": ["Real-time", "Streaming", "Speaker diarization", "API quality"],
                "use_cases": ["Call centers", "Live captioning", "Meeting transcription", "Enterprise"]
            }
        },
        {
            "id": "assemblyai-universal-2",
            "name": "AssemblyAI Universal-2",
            "category": "stt",
            "release_date": "2025-08-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 5,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 commercial STT API, best for long-form content, auto-chapters",
                "why_sota": "Best for podcasts/videos with automatic chapters and summaries",
                "strengths": ["Long-form content", "Auto-chapters", "Summaries", "Easy API"],
                "use_cases": ["Podcasts", "Videos", "Content creators", "Media processing"]
            }
        },

        # =====================================================================
        # MUSIC
        # =====================================================================
        {
            "id": "suno-v4",
            "name": "Suno v4",
            "category": "music",
            "release_date": "2025-09-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 1,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 music generation, best vocal quality and song structure, full songs with lyrics",
                "why_sota": "Most realistic vocals, best at maintaining song structure and coherence",
                "strengths": ["Vocal quality", "Song structure", "Lyrics", "Genre variety"],
                "use_cases": ["Full song creation", "Background music", "Content creators", "Prototyping"]
            }
        },
        {
            "id": "udio",
            "name": "Udio",
            "category": "music",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 2,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 music generation, best for instrumentals and electronic music, high fidelity audio",
                "why_sota": "Superior instrumental quality, better for electronic/EDM genres",
                "strengths": ["Instrumentals", "Electronic music", "Audio fidelity", "Production quality"],
                "use_cases": ["Instrumentals", "EDM", "Background tracks", "Sound design"]
            }
        },
        {
            "id": "musicgen-large",
            "name": "MusicGen Large (Meta)",
            "category": "music",
            "release_date": "2024-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source music, 3.3B params, text/melody to music, runs locally",
                "why_sota": "Best open-source quality, can condition on melody, local deployment",
                "strengths": ["Open-source", "Melody conditioning", "Local deployment", "Good quality"],
                "use_cases": ["Music prototyping", "Melody-based generation", "Research", "Local generation"]
            }
        },
        {
            "id": "stable-audio-2",
            "name": "Stable Audio 2.0 (Stability AI)",
            "category": "music",
            "release_date": "2025-04-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source music, up to 3-min stereo audio, diffusion-based",
                "why_sota": "Longest open-source generation (3 min), good stereo quality",
                "strengths": ["3-min generation", "Stereo output", "Diffusion quality", "Open-source"],
                "use_cases": ["Longer compositions", "Sound effects", "Ambient music", "Audio design"]
            }
        },
        {
            "id": "audiocraft-musicgen-melody",
            "name": "AudioCraft MusicGen Melody (Meta)",
            "category": "music",
            "release_date": "2024-08-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 6,
            "sota_rank_open": 4,
            "metrics": {
                "notes": "#4 open-source music, melody-conditioned generation, AudioCraft family",
                "why_sota": "Best for transforming existing melodies into full arrangements",
                "strengths": ["Melody transformation", "Style transfer", "Arrangement", "Open-source"],
                "use_cases": ["Cover versions", "Style transfer", "Arrangement assistance", "Music education"]
            }
        },
        {
            "id": "yue-7b-icl",
            "name": "YuE (M-A-P/HKUST)",
            "category": "music",
            "release_date": "2025-01-28",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 1,
            "download_url": "https://github.com/multimodal-art-projection/YuE",
            "metrics": {
                "notes": "#1 open-source lyrics-to-song, 7B params, full song generation with vocals, on par with Suno",
                "why_sota": "First open-source lyrics-to-song model matching commercial quality, ICL for style cloning",
                "strengths": ["Full song generation", "Lyrics-to-song", "Style cloning via ICL", "Multi-language", "Apache 2.0"],
                "use_cases": ["Cover songs", "Original music", "Style transfer", "Remixes", "Music production"],
                "vram": "24GB+",
                "models": {
                    "stage1_cot": "m-a-p/YuE-s1-7B-anneal-en-cot",
                    "stage1_icl": "m-a-p/YuE-s1-7B-anneal-en-icl",
                    "stage2": "m-a-p/YuE-s2-1B-general"
                },
                "icl_notes": "Use -icl model with --use_audio_prompt for style cloning from reference audio"
            }
        },

        # =====================================================================
        # VOICE CONVERSION (under TTS umbrella)
        # =====================================================================
        {
            "id": "seed-vc",
            "name": "Seed-VC",
            "category": "tts",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 6,
            "sota_rank_open": 5,
            "download_url": "https://github.com/Plachtaa/seed-vc",
            "metrics": {
                "notes": "SOTA zero-shot voice conversion, dedicated singing voice mode, outperforms RVC in speaker similarity",
                "why_sota": "Best zero-shot voice conversion, singing mode preserves pitch/melody, RTX 5090 compatible",
                "strengths": ["Zero-shot", "Singing voice mode", "400ms latency", "No training needed", "RTX 50-series compatible"],
                "use_cases": ["Voice conversion", "Singing voice conversion", "Cover songs", "Character voices", "Voice cloning"],
                "vram": "8-10GB",
                "vs_rvc": "Outperforms RVC in speaker similarity (SECS) and intelligibility (CER) despite being zero-shot"
            }
        },

        # =====================================================================
        # VIDEO-TO-AUDIO (Foley, SFX, Synchronized Audio from Video)
        # =====================================================================
        {
            "id": "mmaudio-v2-large",
            "name": "MMAudio V2 Large (Sony AI)",
            "category": "video2audio",
            "release_date": "2025-10-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "download_url": "https://github.com/hkchengrex/MMAudio",
            "metrics": {
                "notes": "#1 video-to-audio, 580M params, generates synchronized foley/SFX from video, ~2GB VRAM",
                "why_sota": "Best temporal alignment with video, handles video conditioning directly, lightweight",
                "strengths": ["Temporal sync", "Low VRAM", "Video conditioning", "Foley/SFX quality", "Fast inference"],
                "use_cases": ["Adding sound to silent videos", "Creature foley", "Nature sounds", "Action SFX", "Game audio"],
                "vram": "2GB",
                "parameters": {
                    "steps": {"min": 10, "max": 50, "default": 25},
                    "cfg": {"min": 1.0, "max": 10.0, "default": 4.5},
                    "sample_rate": 44100
                },
                "negative_prompt": "music, speech, narration, human voice, singing"
            }
        },
        {
            "id": "mmaudio-v1",
            "name": "MMAudio V1 (Sony AI)",
            "category": "video2audio",
            "release_date": "2025-05-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "download_url": "https://github.com/hkchengrex/MMAudio",
            "metrics": {
                "notes": "#2 video-to-audio, 150M params, predecessor to V2, still viable for lighter workloads",
                "why_sota": "Smaller model, faster inference, good for prototyping",
                "strengths": ["Very lightweight", "Fast", "Low memory", "Easy to run"],
                "use_cases": ["Quick prototyping", "Batch processing", "Resource-constrained environments"],
                "vram": "<1GB"
            }
        },
        {
            "id": "foleycrafter",
            "name": "FoleyCrafter",
            "category": "video2audio",
            "release_date": "2024-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 video-to-audio, audio adapter approach, good for specific sound effects",
                "why_sota": "Decent quality but less precise temporal alignment than MMAudio",
                "strengths": ["Open-source", "Good for specific SFX"],
                "use_cases": ["Simple foley", "Sound effects", "Basic audio addition"]
            }
        },

        # =====================================================================
        # 3D GENERATION
        # =====================================================================
        {
            "id": "meshy-4",
            "name": "Meshy-4",
            "category": "3d",
            "release_date": "2025-10-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 1,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 3D generation, text/image to 3D mesh, PBR textures, rigging support",
                "why_sota": "Best overall quality and fastest generation, production-ready output",
                "strengths": ["Quality", "Speed", "PBR textures", "Rigging"],
                "use_cases": ["Game assets", "Product visualization", "3D printing", "AR/VR"]
            }
        },
        {
            "id": "tripo-2",
            "name": "Tripo 2.0",
            "category": "3d",
            "release_date": "2025-09-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 2,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#2 3D generation, excellent topology, animation-ready meshes",
                "why_sota": "Best mesh topology for animation, clean geometry",
                "strengths": ["Clean topology", "Animation-ready", "Consistent style"],
                "use_cases": ["Character modeling", "Animation assets", "Game characters"]
            }
        },
        {
            "id": "instantmesh",
            "name": "InstantMesh",
            "category": "3d",
            "release_date": "2025-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 open-source 3D, single image to mesh, runs locally on RTX 3090+",
                "why_sota": "Best open-source quality, fast inference, local deployment",
                "strengths": ["Open-source", "Fast", "Local deployment", "Good quality"],
                "use_cases": ["Prototyping", "Research", "Local generation", "Batch processing"],
                "vram": "24GB"
            }
        },
        {
            "id": "triposr",
            "name": "TripoSR (Stability AI)",
            "category": "3d",
            "release_date": "2024-03-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 open-source 3D, fast single-image to 3D, collaboration with Tripo AI",
                "why_sota": "Fastest open-source image-to-3D, good for quick prototyping",
                "strengths": ["Speed", "Single image input", "Open-source", "Easy to run"],
                "use_cases": ["Quick prototyping", "Concept art to 3D", "Game dev", "Research"]
            }
        },
        {
            "id": "openlrm",
            "name": "OpenLRM",
            "category": "3d",
            "release_date": "2024-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 5,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 open-source 3D, large reconstruction model, good generalization",
                "why_sota": "Best generalization across object types, research-friendly",
                "strengths": ["Generalization", "Research-friendly", "Well-documented", "Active community"],
                "use_cases": ["Research", "Novel objects", "Benchmarking", "Education"]
            }
        },

        # =====================================================================
        # EMBEDDINGS
        # =====================================================================
        {
            "id": "bge-m3",
            "name": "BGE-M3 (BAAI)",
            "category": "embeddings",
            "release_date": "2024-06-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 1,
            "sota_rank_open": 1,
            "metrics": {
                "notes": "#1 embeddings overall, multilingual (100+ langs), hybrid dense+sparse+ColBERT",
                "why_sota": "Best retrieval quality with hybrid approach, works across languages",
                "strengths": ["Multilingual", "Hybrid retrieval", "Dense+Sparse", "ColBERT support"],
                "use_cases": ["RAG systems", "Semantic search", "Multilingual search", "Document retrieval"]
            }
        },
        {
            "id": "gte-qwen2-7b",
            "name": "GTE-Qwen2-7B (Alibaba)",
            "category": "embeddings",
            "release_date": "2025-01-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 2,
            "sota_rank_open": 2,
            "metrics": {
                "notes": "#2 embeddings, 7B param model, best for long documents (8K context)",
                "why_sota": "Highest quality for long document embedding, MTEB leader",
                "strengths": ["Long context", "High quality", "8K tokens", "MTEB leader"],
                "use_cases": ["Long documents", "Legal/medical RAG", "Research papers", "Book search"]
            }
        },
        {
            "id": "e5-mistral-7b",
            "name": "E5-Mistral-7B (Microsoft)",
            "category": "embeddings",
            "release_date": "2024-03-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 3,
            "sota_rank_open": 3,
            "metrics": {
                "notes": "#3 embeddings, instruction-tuned, excellent zero-shot performance",
                "why_sota": "Best zero-shot embedding quality, instruction-following embeddings",
                "strengths": ["Zero-shot", "Instruction-tuned", "Versatile", "Good defaults"],
                "use_cases": ["Zero-shot retrieval", "Custom domains", "Instruction-based search"]
            }
        },
        {
            "id": "jina-embeddings-v3",
            "name": "Jina Embeddings v3",
            "category": "embeddings",
            "release_date": "2025-02-01",
            "is_sota": True,
            "is_open_source": True,
            "sota_rank": 4,
            "sota_rank_open": 4,
            "metrics": {
                "notes": "#4 embeddings, 8K context, late interaction support, task-specific LoRA",
                "why_sota": "Most versatile with task-specific adapters, good long-context support",
                "strengths": ["Task adapters", "8K context", "Versatile", "Late interaction"],
                "use_cases": ["Task-specific retrieval", "Code search", "Multilingual", "Reranking"]
            }
        },
        {
            "id": "cohere-embed-v3",
            "name": "Cohere Embed v3",
            "category": "embeddings",
            "release_date": "2024-11-01",
            "is_sota": True,
            "is_open_source": False,
            "sota_rank": 5,
            "sota_rank_open": None,
            "metrics": {
                "notes": "#1 commercial embeddings API, compression support, 100+ languages",
                "why_sota": "Best commercial API with compression for cost savings",
                "strengths": ["API quality", "Compression", "100+ languages", "Enterprise support"],
                "use_cases": ["Enterprise RAG", "Production search", "Cost-optimized retrieval"]
            }
        },
    ]

    for model in models:
        db.execute("""
            INSERT OR REPLACE INTO models
            (id, name, category, release_date, is_sota, is_open_source, sota_rank, sota_rank_open, metrics, source)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'manual')
        """, (
            model["id"],
            model["name"],
            model["category"],
            model["release_date"],
            model["is_sota"],
            model.get("is_open_source", True),
            model["sota_rank"],
            model.get("sota_rank_open"),
            json.dumps(model.get("metrics", {}))
        ))


def main():
    """Initialize the database."""
    print(f"Initializing database at {DB_PATH}")

    # Remove existing database for fresh start
    if DB_PATH.exists():
        DB_PATH.unlink()
        print("Removed existing database")

    db = sqlite3.connect(str(DB_PATH))

    print("Creating schema...")
    create_schema(db)

    print("Seeding categories...")
    seed_categories(db)

    print("Seeding SOTA models...")
    seed_sota_models(db)

    db.commit()

    # Print stats
    model_count = db.execute("SELECT COUNT(*) FROM models").fetchone()[0]
    open_count = db.execute("SELECT COUNT(*) FROM models WHERE is_open_source = 1").fetchone()[0]
    closed_count = db.execute("SELECT COUNT(*) FROM models WHERE is_open_source = 0").fetchone()[0]

    db.close()

    print(f"\nDatabase initialized successfully!")
    print(f"  Total models: {model_count}")
    print(f"  Open-source: {open_count}")
    print(f"  Closed-source: {closed_count}")
    print(f"  Size: {DB_PATH.stat().st_size / 1024:.1f} KB")


if __name__ == "__main__":
    main()
