# SOTA Tracker

Database of State-of-the-Art AI models with daily auto-updates.

---

## Cyrus Execution Tasks

**For multi-phase overnight work, use the centralized orchestration pattern.**

### Quick Start

1. **Copy template:**
   ```bash
   cp ~/.cyrus/templates/execution_script_template.py scripts/<task_name>_execute.py
   ```

2. **Implement phases:**
   ```python
   class MyExecutor(OrchestrationScript):
       def get_phases(self):
           return {
               "1": self.phase_1_setup,
               "2": self.phase_2_process,
               "3": self.phase_3_verify,
           }
   ```

3. **Create Linear issue from template:**
   ```bash
   cat ~/.cyrus/templates/linear_execution_issue.md
   # Replace <REPO_NAME>, <SCRIPT_NAME>, <ISSUE_ID>
   # Paste into Linear issue description
   ```

4. **Validate before delegating:**
   ```bash
   python ~/.cyrus/scripts/validate_execution_issue.py ROM-XXX
   ```

5. **Delegate to Cyrus** - execution happens automatically

### Resources

- **Template:** `~/.cyrus/templates/execution_script_template.py` (640 lines - complete base class)
- **Docs:** `~/.cyrus/docs/EXECUTION_PATTERN.md` (usage guide)
- **Issue template:** `~/.cyrus/templates/linear_execution_issue.md`
- **Validation:** `~/.cyrus/scripts/validate_execution_issue.py`

### Why Use This Pattern

- Worktree-aware (auto-setup symlinks)
- Crash recovery (checkpoint system)
- Progress tracking (Linear comments)
- Overnight execution (background-safe)
- Proven pattern (ROM-121 reference)

**DON'T:** Create multiple Linear issues with `blockedBy` (doesn't auto-trigger)
**DO:** Single issue, single orchestration script, all phases sequential

---

## Architecture: Data Distribution Pattern

**This repo uses git as a data distribution mechanism.**

```
GitHub Action (daily at 6 AM UTC)
    → Scrapes LMArena, HuggingFace, Artificial Analysis
    → Updates data/sota.db
    → Commits & pushes to GitHub
    → Weekly releases on Sundays
```

## Primary Interface: Static CLAUDE.md Embedding

**MCP is disabled by default.** The recommended approach for Claude Code is static file embedding:

```bash
# Daily auto-update via systemd timer
python ~/scripts/update_sota_claude_md.py --update
```

This generates a compact SOTA summary embedded in `~/.claude/CLAUDE.md`.

**Why static over MCP?**
- **Token cost**: ~721 chars static vs ~8,900 chars MCP tool schemas
- **Simplicity**: No running server, no connection issues
- **Reliability**: Works offline after initial pull

## Alternative Interfaces

| Interface | Use Case | Status |
|-----------|----------|--------|
| Static CLAUDE.md | Claude Code users | **Primary** |
| REST API | Non-Claude clients | Available |
| Direct SQLite | Custom integrations | Available |
| MCP Server | Legacy/special cases | Disabled by default |

## Key Files

| File | Purpose |
|------|---------|
| `init_db.py` | Seeds database with curated SOTA data + forbidden models |
| `rest_api.py` | FastAPI REST server |
| `server.py` | MCP server (optional) |
| `data/sota.db` | SQLite database |

## Adding New Models

1. Edit `init_db.py` - add to `seed_sota_models()`
2. For forbidden/outdated models, add to `seed_forbidden()`
3. Run `python init_db.py` to rebuild local db
4. Commit and push

## Database Schema

**models table:**
- `name`, `category`, `release_date`
- `is_sota`, `is_open_source`
- `sota_rank`, `sota_rank_open`
- `metrics` (JSON: vram, strengths, use_cases, constraints)

**forbidden table:**
- `name`, `reason`, `replacement`, `category`

## Model Constraints

When adding models with specific workflow settings, include constraints:

```python
"metrics": {
    "vram": "19GB",
    "strengths": ["Quality", "Speed"],
    "constraints": {
        "cfg": {"min": 3.0, "max": 5.0, "default": 3.5},
        "shift": {"min": 7.0, "max": 13.0, "default": 7.0}
    }
}
```

## Git Workflow

**Before pushing changes:**
```bash
git pull --rebase  # Get latest scraped data first
```

Conflicts with `sota.db` mean GitHub Action pushed newer data. Take remote:
```bash
git checkout --theirs data/sota.db
```

## What IS Tracked (Intentionally)

| File | Why |
|------|-----|
| `data/sota.db` | Distribution mechanism - users pull for fresh data |
| `data/*_latest.json` | Raw scrape results |
| `data/sota_export.*` | Exports for non-sqlite consumers |

## What Should Be Gitignored

| File | Why |
|------|-----|
| `data/hardware_profiles.json` | User-specific (GPU specs differ per machine) |
