# Technical Backlog

Items identified during code review that were deferred for future implementation.

## Deferred Items

### 1. Async HTTP Requests

**Current State:** All HTTP requests in fetchers use synchronous `urllib.request`. Playwright scrapers use `sync_playwright()`.

**Why Valuable:**
- **Concurrency**: MCP servers handle multiple tool calls. Synchronous HTTP blocks the entire server during network requests (30+ seconds for Playwright scrapes).
- **Responsiveness**: With async, the server can handle other queries while waiting for network I/O.
- **Scalability**: Multiple data sources could be queried in parallel during refresh.

**Why Deferred:**
- Requires refactoring all fetchers to use `aiohttp` instead of `urllib`
- Playwright would need `async_playwright()` which changes the entire scraper architecture
- FastMCP may need async tool handlers
- Current scale (single user, infrequent refreshes) doesn't justify the complexity
- Estimated effort: 2-3 days of refactoring + testing

**When to Implement:** If the MCP server is deployed as a shared service with multiple concurrent users.

---

### 2. Database Connection Pooling

**Current State:** Each function call creates a new SQLite connection via `get_db()`.

**Why Valuable:**
- **Performance**: Connection creation has overhead (~1-2ms per call)
- **Resource Management**: Better control over max connections
- **Prepared Statements**: Pool can cache compiled SQL statements

**Why Deferred:**
- SQLite connections are lightweight (file-based, no network)
- Context managers now ensure connections are closed properly
- Current query volume is low (a few queries per tool call)
- Python's `sqlite3` doesn't have built-in pooling; would need `sqlalchemy` or similar
- Estimated effort: 1 day

**When to Implement:** If profiling shows connection overhead is significant (unlikely for SQLite).

---

### 3. Full Type Hints Coverage

**Current State:** Core functions have type hints, but some helper functions and class methods lack them.

**Why Valuable:**
- **IDE Support**: Better autocomplete and error detection
- **Documentation**: Types serve as inline documentation
- **Static Analysis**: Tools like `mypy` can catch bugs

**Why Deferred:**
- Existing type hints cover the public API
- Internal helpers are small and self-documenting
- Would require ~50 type annotations across 10+ files
- Low impact on runtime behavior
- Estimated effort: 2-4 hours

**When to Implement:** During a dedicated "code quality" sprint or when onboarding new contributors.

---

### 4. Logging Infrastructure

**Current State:** Errors are printed to stdout via `print()`. No structured logging.

**Why Valuable:**
- **Debugging**: Log levels (DEBUG, INFO, WARNING, ERROR) help filter noise
- **Persistence**: Logs can be written to files for post-mortem analysis
- **Observability**: Structured logs can be shipped to monitoring systems

**Why Deferred:**
- MCP servers run as subprocesses; stdout goes to parent
- Current error handling with `print()` is visible in terminal
- Would need to configure logging in each module
- Estimated effort: 1-2 hours

**When to Implement:** If running as a background service or when debugging production issues.

---

### 5. Case-Insensitive Search Index

**Current State:** `check_freshness()` uses `LOWER(name) = LOWER(?)` which prevents index usage.

**Why Valuable:**
- **Performance**: Index lookups are O(log n) vs O(n) full table scan
- **Scalability**: Better performance as model count grows

**Why Deferred:**
- Table has ~150 models; full scan takes <1ms
- Would require schema migration to add `name_lower` column
- SQLite's `COLLATE NOCASE` could work but changes existing data
- Estimated effort: 30 minutes

**When to Implement:** If model count exceeds 1000 and queries become slow.

---

## Completed Improvements (This Release)

- [x] Extracted `is_open_source()` to shared utility (was duplicated 4x)
- [x] Extracted `get_db()` to shared utility (was duplicated 3x)
- [x] Added JSON parse error handling in server.py
- [x] Added input validation for `days` parameter (1-365)
- [x] Created basic test suite (11 tests)

---

## Priority Matrix

| Item | Impact | Effort | Priority |
|------|--------|--------|----------|
| Async HTTP | High | High | P2 |
| Connection Pooling | Low | Medium | P4 |
| Type Hints | Low | Low | P3 |
| Logging | Medium | Low | P3 |
| Case-Insensitive Index | Low | Low | P4 |

**Legend:**
- P1: Do immediately
- P2: Do when scaling becomes necessary
- P3: Nice to have, do during maintenance
- P4: Only if profiling shows it's needed
