"""Tests for database migration system."""

import sqlite3
import tempfile
from pathlib import Path
import sys

sys.path.insert(0, str(Path(__file__).parent.parent))

from migrations.migrate import (
    MIGRATIONS,
    ensure_version_table,
    get_current_version,
    get_pending_migrations,
    apply_migration,
)


class TestMigrations:
    """Tests for migration utilities."""

    def test_migrations_have_valid_structure(self):
        """Each migration should have (version, name, up_sql, down_sql)."""
        for migration in MIGRATIONS:
            assert len(migration) == 4
            version, name, up_sql, down_sql = migration
            assert isinstance(version, int)
            assert isinstance(name, str)
            assert isinstance(up_sql, str)
            assert isinstance(down_sql, str)

    def test_migrations_are_sequential(self):
        """Migration versions should be sequential starting from 1."""
        versions = [m[0] for m in MIGRATIONS]
        expected = list(range(1, len(MIGRATIONS) + 1))
        assert versions == expected

    def test_migration_names_are_unique(self):
        """Migration names should be unique."""
        names = [m[1] for m in MIGRATIONS]
        assert len(names) == len(set(names))


class TestVersionTable:
    """Tests for schema version tracking."""

    def test_ensure_version_table_creates_table(self):
        """Should create schema_version table if not exists."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as f:
            db = sqlite3.connect(f.name)
            db.row_factory = sqlite3.Row

            ensure_version_table(db)

            # Table should exist
            result = db.execute(
                "SELECT name FROM sqlite_master WHERE type='table' AND name='schema_version'"
            ).fetchone()
            assert result is not None

            db.close()

    def test_get_current_version_empty_db(self):
        """Should return 0 for empty database."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as f:
            db = sqlite3.connect(f.name)
            db.row_factory = sqlite3.Row

            ensure_version_table(db)
            version = get_current_version(db)

            assert version == 0

            db.close()

    def test_get_pending_migrations(self):
        """Should return all migrations for fresh database."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as f:
            db = sqlite3.connect(f.name)
            db.row_factory = sqlite3.Row

            ensure_version_table(db)
            pending = get_pending_migrations(db)

            # All migrations should be pending
            assert len(pending) == len(MIGRATIONS)

            db.close()


class TestApplyMigration:
    """Tests for applying migrations."""

    def test_apply_simple_migration(self):
        """Should apply a simple migration and record version."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as f:
            db = sqlite3.connect(f.name)
            db.row_factory = sqlite3.Row

            ensure_version_table(db)

            # Apply first migration
            version, name, up_sql, _ = MIGRATIONS[0]
            apply_migration(db, version, name, up_sql)

            # Version should be recorded
            current = get_current_version(db)
            assert current == version

            db.close()

    def test_migrations_are_idempotent(self):
        """Should handle re-running migrations gracefully."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as f:
            db = sqlite3.connect(f.name)
            db.row_factory = sqlite3.Row

            ensure_version_table(db)

            # Apply same migration twice - should not error
            version, name, up_sql, _ = MIGRATIONS[0]
            apply_migration(db, version, name, up_sql)
            apply_migration(db, version, name, up_sql)  # Should not raise

            db.close()


# Run with: python -m pytest tests/test_migrations.py -v
