"""Tests for edge cases and error handling."""

import pytest
import sys
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from utils.models import normalize_model_id, build_model_dict
from utils.hardware import parse_vram_string, vram_fits, configure_profile


class TestModelNormalization:
    """Tests for model ID normalization edge cases."""

    def test_special_characters(self):
        """Test normalization with slashes, dots, and spaces."""
        assert normalize_model_id("GPT-4.5 Turbo/Pro") == "gpt-4-5-turbo-pro"

    def test_empty_string(self):
        """Test normalization of empty string."""
        assert normalize_model_id("") == ""

    def test_already_normalized(self):
        """Test that already normalized IDs are unchanged."""
        assert normalize_model_id("gpt-4-turbo") == "gpt-4-turbo"

    def test_multiple_spaces(self):
        """Test handling of multiple consecutive spaces."""
        assert normalize_model_id("Model  Name") == "model--name"

    def test_leading_trailing_spaces(self):
        """Test that leading/trailing spaces become hyphens."""
        assert normalize_model_id(" Model ") == "-model-"

    def test_mixed_case(self):
        """Test that mixed case is lowercased."""
        assert normalize_model_id("FLUX.1-Dev") == "flux-1-dev"


class TestBuildModelDict:
    """Tests for model dictionary builder."""

    def test_basic_model(self):
        """Test basic model creation."""
        model = build_model_dict(
            name="Test Model",
            rank=1,
            category="llm_api",
            is_open_source=True
        )
        assert model["id"] == "test-model"
        assert model["name"] == "Test Model"
        assert model["sota_rank"] == 1
        assert model["category"] == "llm_api"
        assert model["is_open_source"] is True
        assert model["metrics"] == {}

    def test_with_metrics(self):
        """Test model creation with metrics."""
        model = build_model_dict(
            name="Test Model",
            rank=1,
            category="llm_api",
            is_open_source=True,
            metrics={"elo": 1500, "source": "test"}
        )
        assert model["metrics"]["elo"] == 1500
        assert model["metrics"]["source"] == "test"

    def test_with_extra_fields(self):
        """Test model creation with extra fields."""
        model = build_model_dict(
            name="Test Model",
            rank=1,
            category="llm_api",
            is_open_source=True,
            release_date="2025-01-01",
            download_url="https://example.com"
        )
        assert model["release_date"] == "2025-01-01"
        assert model["download_url"] == "https://example.com"


class TestVramParsing:
    """Tests for VRAM string parsing edge cases."""

    def test_unusual_formats(self):
        """Test parsing of unusual VRAM string formats."""
        assert parse_vram_string("24 GB GPU GGUF F32") == 24
        assert parse_vram_string("~16GB") == 16

    def test_lowercase_gb(self):
        """Test parsing with lowercase 'gb'."""
        assert parse_vram_string("8gb") == 8

    def test_no_space(self):
        """Test parsing without space before GB."""
        assert parse_vram_string("32GB") == 32

    def test_with_extra_text(self):
        """Test parsing with extra text around the number."""
        assert parse_vram_string("requires 24 GB VRAM") == 24

    def test_none_value(self):
        """Test parsing None value."""
        assert parse_vram_string(None) is None

    def test_invalid_string(self):
        """Test parsing invalid string returns None."""
        assert parse_vram_string("no numbers here") is None

    def test_integer_input(self):
        """Test parsing integer input."""
        assert parse_vram_string(16) == 16


class TestVramFits:
    """Tests for VRAM fitting logic."""

    def test_model_fits_with_headroom(self):
        """Test model fits with plenty of headroom."""
        assert vram_fits(16, 32) is True

    def test_model_exact_fit(self):
        """Test model fits exactly."""
        assert vram_fits(32, 32) is True

    def test_model_does_not_fit(self):
        """Test model does not fit."""
        assert vram_fits(48, 32) is False

    def test_none_model_vram(self):
        """Test None model VRAM always fits."""
        assert vram_fits(None, 32) is True

    def test_zero_model_vram(self):
        """Test zero model VRAM always fits."""
        assert vram_fits(0, 32) is True


class TestHardwareValidation:
    """Tests for hardware profile validation."""

    def test_negative_vram_rejected(self):
        """Test that negative VRAM is rejected."""
        with pytest.raises(ValueError, match="vram_gb must be a positive integer"):
            configure_profile(vram_gb=-1)

    def test_zero_vram_rejected(self):
        """Test that zero VRAM is rejected."""
        with pytest.raises(ValueError, match="vram_gb must be a positive integer"):
            configure_profile(vram_gb=0)

    def test_negative_ram_rejected(self):
        """Test that negative RAM is rejected."""
        with pytest.raises(ValueError, match="ram_gb must be a positive integer"):
            configure_profile(ram_gb=-16)

    def test_zero_cpu_threads_rejected(self):
        """Test that zero CPU threads is rejected."""
        with pytest.raises(ValueError, match="cpu_threads must be a positive integer"):
            configure_profile(cpu_threads=0)

    def test_valid_config(self):
        """Test valid configuration succeeds."""
        result = configure_profile(vram_gb=32, ram_gb=64, cpu_threads=16)
        assert "vram_gb" in result
        assert result["vram_gb"] == 32
