"""
LMArena (Chatbot Arena) scraper using Playwright.

Scrapes Elo rankings from https://lmarena.ai/leaderboard

The page uses JavaScript to render a dynamic table, so we need
a real browser to extract the data.
"""

import json
import re
import sys
from datetime import datetime
from pathlib import Path
from typing import Optional
from playwright.sync_api import sync_playwright, Page, TimeoutError as PlaywrightTimeout

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
from utils.classification import is_open_source

# Project paths (use absolute paths to prevent path traversal)
PROJECT_DIR = Path(__file__).parent.parent
DATA_DIR = PROJECT_DIR / "data"


class LMArenaScraper:
    """Scrape Elo rankings from LMArena."""

    URL = "https://lmarena.ai/leaderboard"
    TIMEOUT = 30000  # 30 seconds

    def __init__(self, headless: bool = True):
        self.headless = headless

    def scrape(self) -> dict:
        """
        Scrape the LMArena leaderboard.

        Returns:
            {
                "source": "lmarena",
                "url": "https://lmarena.ai/leaderboard",
                "scraped_at": "2026-01-09T...",
                "models": [
                    {"rank": 1, "name": "...", "elo": 1234, "is_open_source": True, ...},
                    ...
                ]
            }
        """
        with sync_playwright() as p:
            browser = p.chromium.launch(headless=self.headless)
            try:
                page = browser.new_page()
                page.set_default_timeout(self.TIMEOUT)

                print(f"Navigating to {self.URL}...")
                page.goto(self.URL, wait_until="networkidle")

                # Wait for the leaderboard table to appear
                print("Waiting for leaderboard table...")
                page.wait_for_selector("table", timeout=self.TIMEOUT)

                # Give it a moment for data to fully load
                page.wait_for_timeout(2000)

                # Extract data from the table
                models = self._extract_table_data(page)

                return {
                    "source": "lmarena",
                    "url": self.URL,
                    "scraped_at": datetime.now().isoformat(),
                    "model_count": len(models),
                    "models": models
                }

            except PlaywrightTimeout as e:
                print(f"Timeout waiting for page: {e}")
                return {"source": "lmarena", "error": str(e), "models": []}
            except Exception as e:
                print(f"Scraping error: {e}")
                return {"source": "lmarena", "error": str(e), "models": []}
            finally:
                browser.close()

    def _extract_table_data(self, page: Page) -> list[dict]:
        """Extract model data from the leaderboard table."""
        models = []

        # Try to find table rows
        rows = page.query_selector_all("table tbody tr")

        if not rows:
            # Alternative: Try to find data in a different structure
            print("No table rows found, trying alternative selectors...")
            rows = page.query_selector_all("[data-testid='leaderboard-row']")

        if not rows:
            # Last resort: Extract from page content
            print("Using content extraction fallback...")
            return self._extract_from_content(page)

        print(f"Found {len(rows)} rows in table")

        for idx, row in enumerate(rows[:50]):  # Top 50 models
            try:
                cells = row.query_selector_all("td")
                if len(cells) >= 2:
                    # Extract text from cells
                    rank_text = cells[0].inner_text().strip() if len(cells) > 0 else str(idx + 1)
                    name = cells[1].inner_text().strip() if len(cells) > 1 else ""
                    elo_text = cells[2].inner_text().strip() if len(cells) > 2 else ""

                    # Parse rank
                    rank_match = re.search(r'\d+', rank_text)
                    rank = int(rank_match.group()) if rank_match else idx + 1

                    # Parse Elo
                    elo_match = re.search(r'[\d,]+', elo_text.replace(',', ''))
                    elo = int(elo_match.group()) if elo_match else None

                    if name:
                        models.append({
                            "rank": rank,
                            "name": name,
                            "elo": elo,
                            "is_open_source": is_open_source(name),
                            "category": "llm_api",
                        })
            except Exception as e:
                print(f"Error parsing row {idx}: {e}")
                continue

        return models

    def _extract_from_content(self, page: Page) -> list[dict]:
        """Fallback: Extract data from page content using regex."""
        models = []
        content = page.content()

        # Limit content length to prevent ReDoS attacks (10MB max)
        if len(content) > 10 * 1024 * 1024:
            content = content[:10 * 1024 * 1024]

        # Look for patterns like "1. Model Name 1234" or JSON-like structures
        # This is a best-effort fallback

        # Try to find JSON data in script tags
        # Use [^<]+ instead of .*? to prevent catastrophic backtracking
        json_match = re.search(r'<script[^>]*type="application/json"[^>]*>([^<]+)</script>', content)
        if json_match:
            try:
                data = json.loads(json_match.group(1))
                # Navigate to find model data
                if isinstance(data, dict):
                    for key in ["models", "leaderboard", "data", "props"]:
                        if key in data:
                            items = data[key]
                            if isinstance(items, list):
                                for idx, item in enumerate(items[:50]):
                                    if isinstance(item, dict):
                                        models.append({
                                            "rank": item.get("rank", idx + 1),
                                            "name": item.get("name", item.get("model", "")),
                                            "elo": item.get("elo", item.get("rating")),
                                            "is_open_source": self._is_open_source(item.get("name", "")),
                                            "category": "llm_api",
                                        })
            except json.JSONDecodeError:
                pass

        # If no JSON, try regex patterns
        if not models:
            # Pattern: rank, name, elo score
            pattern = r'(\d+)\s*[.)\s]+([A-Za-z0-9\-\._/ ]+?)\s+(\d{3,4})'
            matches = re.findall(pattern, content)
            for rank, name, elo in matches[:50]:
                name = name.strip()
                if len(name) > 2 and not name.isdigit():
                    models.append({
                        "rank": int(rank),
                        "name": name,
                        "elo": int(elo),
                        "is_open_source": is_open_source(name),
                        "category": "llm_api",
                    })

        return models



def scrape_lmarena() -> dict:
    """Convenience function to scrape LMArena."""
    scraper = LMArenaScraper()
    return scraper.scrape()


# CLI test
if __name__ == "__main__":
    print("Scraping LMArena leaderboard...")
    result = scrape_lmarena()

    if result.get("error"):
        print(f"Error: {result['error']}")
    else:
        print(f"\nScraped {result['model_count']} models at {result['scraped_at']}")
        print("\nTop 10:")
        for m in result["models"][:10]:
            badge = "" if m["is_open_source"] else " [CLOSED]"
            elo = m.get("elo", "N/A")
            print(f"  #{m['rank']} {m['name']}{badge}: Elo {elo}")

        # Save to JSON (use absolute path to prevent path traversal)
        output_path = DATA_DIR / "lmarena_latest.json"
        with open(output_path, "w") as f:
            json.dump(result, f, indent=2)
        print(f"\nSaved to {output_path}")
