#!/usr/bin/env python3
"""
Civitai API scraper for image generation models.

Civitai is the largest repository of Stable Diffusion models, LoRAs, and other
image generation assets. This scraper fetches top-rated checkpoints.

API Reference: https://github.com/civitai/civitai/wiki/REST-API-Reference
Rate limit: ~200 requests/hour (no auth required)

Usage:
    from scrapers.civitai import CivitaiScraper
    scraper = CivitaiScraper()
    result = scraper.scrape(model_type="Checkpoint", limit=20)
"""

import json
import urllib.request
from datetime import datetime
from pathlib import Path

PROJECT_DIR = Path(__file__).parent.parent
DATA_DIR = PROJECT_DIR / "data"


class CivitaiScraper:
    """Scraper for Civitai model rankings."""

    BASE_URL = "https://civitai.com/api/v1"

    def __init__(self):
        self.headers = {
            "User-Agent": "SOTA-Tracker/1.0 (https://github.com/sota-tracker-mcp)"
        }

    def scrape(self, model_type: str = "Checkpoint", limit: int = 20) -> dict:
        """
        Fetch top models from Civitai.

        Args:
            model_type: Model type to fetch. Options:
                - "Checkpoint": Full SD models (SDXL, SD 1.5, Flux, etc.)
                - "LORA": LoRA adapters
                - "TextualInversion": Textual inversion embeddings
                - "Hypernetwork": Hypernetworks
                - "AestheticGradient": Aesthetic gradients
                - "Controlnet": ControlNet models
            limit: Number of models to fetch (max 100)

        Returns:
            Dict with models list and metadata
        """
        # Build API URL
        # Sort by "Highest Rated" for quality, could also use "Most Downloaded"
        url = (
            f"{self.BASE_URL}/models"
            f"?limit={min(limit, 100)}"
            f"&sort=Highest%20Rated"
            f"&types={model_type}"
            f"&nsfw=false"  # Filter out NSFW by default
        )

        try:
            req = urllib.request.Request(url, headers=self.headers)
            with urllib.request.urlopen(req, timeout=30) as response:
                data = json.loads(response.read().decode())
        except urllib.error.URLError as e:
            return {"error": f"Network error: {e}", "models": []}
        except json.JSONDecodeError as e:
            return {"error": f"JSON decode error: {e}", "models": []}
        except Exception as e:
            return {"error": str(e), "models": []}

        models = []
        for item in data.get("items", []):
            stats = item.get("stats", {})

            # Extract base model info (SDXL, SD 1.5, Flux, etc.)
            model_versions = item.get("modelVersions", [])
            base_model = None
            if model_versions:
                base_model = model_versions[0].get("baseModel")

            models.append({
                "rank": len(models) + 1,
                "name": item.get("name"),
                "id": f"civitai-{item.get('id')}",
                "category": "image_gen",
                "is_open_source": True,  # Civitai models are open
                "metrics": {
                    "rating": stats.get("rating"),
                    "rating_count": stats.get("ratingCount"),
                    "downloads": stats.get("downloadCount"),
                    "favorites": stats.get("favoriteCount"),
                    "comments": stats.get("commentCount"),
                    "base_model": base_model,
                    "model_type": model_type,
                    "source": "civitai",
                    "civitai_id": item.get("id"),
                    "civitai_url": f"https://civitai.com/models/{item.get('id')}"
                }
            })

        result = {
            "source": "civitai",
            "url": url,
            "scraped_at": datetime.now().isoformat(),
            "model_count": len(models),
            "model_type": model_type,
            "models": models
        }

        # Save raw data for debugging/inspection
        output_path = DATA_DIR / "civitai_latest.json"
        try:
            with open(output_path, "w") as f:
                json.dump(result, f, indent=2)
        except Exception:
            pass  # Don't fail if we can't save cache

        return result

    def scrape_all_types(self, limit_per_type: int = 10) -> dict:
        """
        Scrape top models across multiple types.

        Returns combined results for Checkpoints, LoRAs, and ControlNets.
        """
        all_models = []
        results = {}

        for model_type in ["Checkpoint", "LORA", "Controlnet"]:
            result = self.scrape(model_type=model_type, limit=limit_per_type)
            results[model_type] = result
            if result.get("models"):
                all_models.extend(result["models"])

        return {
            "source": "civitai",
            "scraped_at": datetime.now().isoformat(),
            "model_count": len(all_models),
            "models": all_models,
            "by_type": results
        }


# CLI for testing
if __name__ == "__main__":
    import sys

    scraper = CivitaiScraper()

    if len(sys.argv) > 1 and sys.argv[1] == "--all":
        result = scraper.scrape_all_types(limit_per_type=5)
    else:
        result = scraper.scrape(model_type="Checkpoint", limit=10)

    print(json.dumps(result, indent=2))
