"""
Artificial Analysis scraper using Playwright.

Scrapes quality benchmarks from:
- https://artificialanalysis.ai/leaderboards/models (LLMs)
- https://artificialanalysis.ai/text-to-image/arena (Image Gen)
- https://artificialanalysis.ai/text-to-video/arena (Video Gen)

These pages use JavaScript rendering, requiring a real browser.
"""

import json
import re
import sys
from datetime import datetime
from pathlib import Path
from typing import Optional
from playwright.sync_api import sync_playwright, Page, TimeoutError as PlaywrightTimeout

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
from utils.classification import is_open_source

# Project paths (use absolute paths to prevent path traversal)
PROJECT_DIR = Path(__file__).parent.parent
DATA_DIR = PROJECT_DIR / "data"


class ArtificialAnalysisScraper:
    """Scrape SOTA data from Artificial Analysis."""

    BASE_URL = "https://artificialanalysis.ai"

    ENDPOINTS = {
        "llm": "/leaderboards/models",
        "image_gen": "/image/arena",
        "video": "/video/arena",
        "tts": "/speech/arena",
    }

    TIMEOUT = 30000  # 30 seconds

    def __init__(self, headless: bool = True):
        self.headless = headless

    def scrape(self, category: str = "llm") -> dict:
        """
        Scrape a specific category from Artificial Analysis.

        Args:
            category: One of "llm", "image_gen", "video", "tts"

        Returns:
            {
                "source": "artificial_analysis",
                "category": "llm",
                "url": "...",
                "scraped_at": "...",
                "models": [...]
            }
        """
        endpoint = self.ENDPOINTS.get(category)
        if not endpoint:
            return {
                "source": "artificial_analysis",
                "category": category,
                "error": f"Unknown category: {category}",
                "models": []
            }

        url = f"{self.BASE_URL}{endpoint}"

        with sync_playwright() as p:
            browser = p.chromium.launch(headless=self.headless)
            try:
                page = browser.new_page()
                page.set_default_timeout(self.TIMEOUT)

                print(f"Navigating to {url}...")
                page.goto(url, wait_until="domcontentloaded")

                # Wait for content to load
                print("Waiting for content to load...")
                page.wait_for_timeout(3000)  # Give JS time to render

                # Extract data based on category
                if category == "llm":
                    models = self._extract_llm_data(page)
                else:
                    models = self._extract_arena_data(page, category)

                return {
                    "source": "artificial_analysis",
                    "category": category,
                    "url": url,
                    "scraped_at": datetime.now().isoformat(),
                    "model_count": len(models),
                    "models": models
                }

            except PlaywrightTimeout as e:
                print(f"Timeout: {e}")
                return {"source": "artificial_analysis", "category": category, "error": str(e), "models": []}
            except Exception as e:
                print(f"Error: {e}")
                return {"source": "artificial_analysis", "category": category, "error": str(e), "models": []}
            finally:
                browser.close()

    def scrape_all(self) -> dict:
        """Scrape all categories."""
        results = {}
        for category in self.ENDPOINTS.keys():
            print(f"\n=== Scraping {category} ===")
            results[category] = self.scrape(category)
        return results

    def _extract_llm_data(self, page: Page) -> list[dict]:
        """Extract LLM leaderboard data."""
        models = []

        # Try to find the leaderboard table
        rows = page.query_selector_all("table tbody tr")

        if rows:
            print(f"Found {len(rows)} rows in LLM table")
            for idx, row in enumerate(rows[:30]):
                try:
                    cells = row.query_selector_all("td")
                    if len(cells) >= 3:
                        # Extract model info
                        name_cell = cells[0].inner_text().strip()
                        # Parse name (might include provider)
                        name = name_cell.split('\n')[0].strip()

                        # Try to find Elo/score
                        score = None
                        for cell in cells[1:]:
                            text = cell.inner_text().strip()
                            score_match = re.search(r'[\d,]+', text.replace(',', ''))
                            if score_match:
                                val = int(score_match.group())
                                if 500 < val < 2000:  # Likely Elo score
                                    score = val
                                    break

                        if name:
                            models.append({
                                "rank": idx + 1,
                                "name": name,
                                "elo": score,
                                "is_open_source": is_open_source(name),
                                "category": "llm_api",
                            })
                except Exception as e:
                    print(f"Error parsing row {idx}: {e}")

        # Fallback: Try to extract from page content
        if not models:
            models = self._extract_from_content(page, "llm")

        return models

    def _extract_arena_data(self, page: Page, category: str) -> list[dict]:
        """Extract arena data for image/video/tts."""
        models = []

        # Arena pages often have a different structure
        # Try multiple selectors
        selectors = [
            "table tbody tr",
            "[data-testid='leaderboard-row']",
            ".leaderboard-item",
            "[class*='model-row']",
        ]

        for selector in selectors:
            rows = page.query_selector_all(selector)
            if rows:
                print(f"Found {len(rows)} items with selector: {selector}")
                break

        if rows:
            for idx, row in enumerate(rows[:20]):
                try:
                    text = row.inner_text().strip()
                    lines = text.split('\n')

                    # Try to parse name and score
                    name = lines[0].strip() if lines else ""

                    score = None
                    for line in lines:
                        score_match = re.search(r'(\d{3,4})', line)
                        if score_match:
                            score = int(score_match.group(1))
                            break

                    if name and len(name) > 2:
                        models.append({
                            "rank": idx + 1,
                            "name": name,
                            "elo": score,
                            "is_open_source": is_open_source(name),
                            "category": self._map_category(category),
                        })
                except Exception as e:
                    print(f"Error parsing item {idx}: {e}")

        # Fallback
        if not models:
            models = self._extract_from_content(page, category)

        return models

    def _extract_from_content(self, page: Page, category: str) -> list[dict]:
        """Fallback: Extract from page content."""
        models = []
        content = page.content()

        # Limit content length to prevent ReDoS attacks (10MB max)
        if len(content) > 10 * 1024 * 1024:
            content = content[:10 * 1024 * 1024]

        # Try to find Next.js data
        # Use [^<]+ instead of .*? to prevent catastrophic backtracking
        next_data_match = re.search(r'<script id="__NEXT_DATA__" type="application/json">([^<]+)</script>', content)
        if next_data_match:
            try:
                data = json.loads(next_data_match.group(1))
                # Navigate to find model data
                props = data.get("props", {}).get("pageProps", {})

                # Look for model lists
                for key in ["models", "leaderboard", "rankings", "data"]:
                    if key in props and isinstance(props[key], list):
                        for idx, item in enumerate(props[key][:30]):
                            if isinstance(item, dict):
                                name = item.get("name", item.get("model", item.get("title", "")))
                                if name:
                                    models.append({
                                        "rank": item.get("rank", idx + 1),
                                        "name": name,
                                        "elo": item.get("elo", item.get("score", item.get("rating"))),
                                        "is_open_source": is_open_source(name),
                                        "category": self._map_category(category),
                                    })
                        break
            except json.JSONDecodeError:
                pass

        return models

    def _map_category(self, aa_category: str) -> str:
        """Map scraper category to our internal category."""
        mapping = {
            "llm": "llm_api",
            "image_gen": "image_gen",
            "video": "video",
            "tts": "tts",
        }
        return mapping.get(aa_category, aa_category)



def scrape_artificial_analysis(category: str = "llm") -> dict:
    """Convenience function to scrape Artificial Analysis."""
    scraper = ArtificialAnalysisScraper()
    return scraper.scrape(category)


# CLI test
if __name__ == "__main__":
    import sys

    category = sys.argv[1] if len(sys.argv) > 1 else "llm"

    print(f"Scraping Artificial Analysis ({category})...")
    result = scrape_artificial_analysis(category)

    if result.get("error"):
        print(f"Error: {result['error']}")
    else:
        print(f"\nScraped {result['model_count']} models at {result['scraped_at']}")
        print("\nTop 10:")
        for m in result["models"][:10]:
            badge = "" if m["is_open_source"] else " [CLOSED]"
            elo = m.get("elo", "N/A")
            print(f"  #{m['rank']} {m['name']}{badge}: Score {elo}")

        # Save to JSON (use absolute path to prevent path traversal)
        output_path = DATA_DIR / f"aa_{category}_latest.json"
        with open(output_path, "w") as f:
            json.dump(result, f, indent=2)
        print(f"\nSaved to {output_path}")
