#!/usr/bin/env python3
"""
Database migration system for SOTA Tracker.

Handles schema upgrades gracefully without data loss.

Usage:
    python migrations/migrate.py           # Apply all pending migrations
    python migrations/migrate.py --status  # Show migration status
    python migrations/migrate.py --reset   # Reset and reinitialize (DESTRUCTIVE)
"""

import argparse
import json
import sqlite3
import sys
from datetime import datetime
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
from constants import DB_PATH
from utils.db import get_db_context

# Migration definitions: (version, name, up_sql, down_sql)
MIGRATIONS = [
    (1, "initial_schema", """
        -- This is the base schema, applied by init_db.py
        -- Kept here for reference and version tracking
        CREATE TABLE IF NOT EXISTS schema_version (
            version INTEGER PRIMARY KEY,
            name TEXT NOT NULL,
            applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        );
    """, "DROP TABLE IF EXISTS schema_version;"),

    (2, "add_download_url", """
        -- Add download_url column to models table for direct HuggingFace links
        ALTER TABLE models ADD COLUMN download_url TEXT;
    """, """
        -- SQLite doesn't support DROP COLUMN, so we'd need to recreate table
        -- For simplicity, this is a no-op in down migration
        SELECT 1;
    """),

    (3, "add_pricing_columns", """
        -- Add pricing columns for API models
        ALTER TABLE models ADD COLUMN price_per_1m_input REAL;
        ALTER TABLE models ADD COLUMN price_per_1m_output REAL;
    """, "SELECT 1;"),

    (4, "add_benchmark_scores", """
        -- Add common benchmark score columns
        ALTER TABLE models ADD COLUMN elo_score INTEGER;
        ALTER TABLE models ADD COLUMN humaneval_score REAL;
        ALTER TABLE models ADD COLUMN swebench_score REAL;
    """, "SELECT 1;"),
]


def ensure_version_table(db: sqlite3.Connection):
    """Ensure schema_version table exists."""
    db.execute("""
        CREATE TABLE IF NOT EXISTS schema_version (
            version INTEGER PRIMARY KEY,
            name TEXT NOT NULL,
            applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)
    db.commit()


def get_current_version(db: sqlite3.Connection) -> int:
    """Get current schema version."""
    try:
        row = db.execute("SELECT MAX(version) as v FROM schema_version").fetchone()
        return row["v"] or 0
    except sqlite3.OperationalError:
        return 0


def get_pending_migrations(db: sqlite3.Connection) -> list:
    """Get list of migrations that haven't been applied."""
    current = get_current_version(db)
    return [(v, n, up, down) for v, n, up, down in MIGRATIONS if v > current]


def apply_migration(db: sqlite3.Connection, version: int, name: str, sql: str):
    """Apply a single migration."""
    print(f"  Applying migration {version}: {name}")

    # Split SQL into statements and execute each
    for statement in sql.strip().split(";"):
        statement = statement.strip()
        if statement and not statement.startswith("--"):
            try:
                db.execute(statement)
            except sqlite3.OperationalError as e:
                # Ignore "duplicate column" errors for idempotency
                if "duplicate column" not in str(e).lower():
                    raise

    # Record migration
    db.execute(
        "INSERT OR REPLACE INTO schema_version (version, name) VALUES (?, ?)",
        (version, name)
    )
    db.commit()


def migrate():
    """Apply all pending migrations."""
    if not DB_PATH.exists():
        print(f"Database not found at {DB_PATH}")
        print("Run 'python init_db.py' first to create the database.")
        return False

    with get_db_context(DB_PATH) as db:
        ensure_version_table(db)

        pending = get_pending_migrations(db)
        if not pending:
            print("Database is up to date.")
            return True

        print(f"Applying {len(pending)} migration(s)...")

        for version, name, up_sql, _ in pending:
            apply_migration(db, version, name, up_sql)

        print(f"Done. Database is now at version {get_current_version(db)}.")
    return True


def status():
    """Show migration status."""
    if not DB_PATH.exists():
        print("Database not found.")
        return

    with get_db_context(DB_PATH) as db:
        ensure_version_table(db)

        current = get_current_version(db)
        pending = get_pending_migrations(db)

        print(f"Current version: {current}")
        print(f"Latest version:  {MIGRATIONS[-1][0]}")
        print()

        # Show applied migrations
        applied = db.execute(
            "SELECT version, name, applied_at FROM schema_version ORDER BY version"
        ).fetchall()

        if applied:
            print("Applied migrations:")
            for row in applied:
                print(f"  v{row['version']}: {row['name']} ({row['applied_at']})")
        else:
            print("No migrations applied yet.")

        if pending:
            print(f"\nPending migrations ({len(pending)}):")
            for version, name, _, _ in pending:
                print(f"  v{version}: {name}")
        else:
            print("\nNo pending migrations.")


def reset():
    """Reset database (DESTRUCTIVE)."""
    if DB_PATH.exists():
        print(f"Removing {DB_PATH}...")
        DB_PATH.unlink()
    print("Database reset. Run 'python init_db.py' to reinitialize.")


def main():
    parser = argparse.ArgumentParser(description="SOTA Tracker database migrations")
    parser.add_argument("--status", action="store_true", help="Show migration status")
    parser.add_argument("--reset", action="store_true", help="Reset database (DESTRUCTIVE)")
    args = parser.parse_args()

    if args.status:
        status()
    elif args.reset:
        confirm = input("This will DELETE all data. Type 'yes' to confirm: ")
        if confirm.lower() == "yes":
            reset()
        else:
            print("Aborted.")
    else:
        migrate()


if __name__ == "__main__":
    main()
