"""
LMArena (Chatbot Arena) fetcher.

Fetches Elo rankings from LMSYS Chatbot Arena.
Data is available via HuggingFace Spaces (free).

Source: https://huggingface.co/spaces/lmarena-ai/chatbot-arena-leaderboard
"""

import json
import sys
import urllib.request
import urllib.error
from pathlib import Path
from typing import Optional

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
from constants import HTTP_TIMEOUT_SECONDS, MAX_LEADERBOARD_SIZE
from utils.classification import is_open_source
from utils.models import normalize_model_id, build_model_dict
from utils.log import get_logger

logger = get_logger("fetchers.lmarena")


class LMArenaFetcher:
    """Fetch Elo rankings from Chatbot Arena."""

    # Leaderboard data hosted on HuggingFace Spaces
    LEADERBOARD_URL = "https://huggingface.co/spaces/lmarena-ai/chatbot-arena-leaderboard/resolve/main/leaderboard_table_20240701.csv"

    # Alternative: JSON endpoint (if available)
    JSON_URL = "https://raw.githubusercontent.com/lm-sys/FastChat/main/fastchat/serve/leaderboard/elo_results.json"

    def __init__(self, timeout: int = HTTP_TIMEOUT_SECONDS):
        self.timeout = timeout

    def fetch_elo_rankings(self) -> list[dict]:
        """
        Fetch current Elo rankings from Chatbot Arena.

        Returns list of models with Elo scores, sorted by rank.
        """
        # Try JSON first (more reliable)
        data = self._fetch_json()
        if data:
            return data

        # Fallback to CSV
        return self._fetch_csv()

    def _fetch_json(self) -> list[dict]:
        """Try to fetch from JSON source."""
        try:
            req = urllib.request.Request(
                self.JSON_URL,
                headers={"User-Agent": "sota-tracker-mcp/1.0"}
            )
            with urllib.request.urlopen(req, timeout=self.timeout) as resp:
                data = json.loads(resp.read().decode())
                return self._parse_json(data)
        except Exception as e:
            logger.warning(f"LMArena JSON fetch failed: {e}")
            return []

    def _fetch_csv(self) -> list[dict]:
        """Fallback to CSV source."""
        try:
            req = urllib.request.Request(
                self.LEADERBOARD_URL,
                headers={"User-Agent": "sota-tracker-mcp/1.0"}
            )
            with urllib.request.urlopen(req, timeout=self.timeout) as resp:
                csv_text = resp.read().decode()
                return self._parse_csv(csv_text)
        except Exception as e:
            logger.warning(f"LMArena CSV fetch failed: {e}")
            return []

    def _parse_json(self, data: dict) -> list[dict]:
        """Parse JSON Elo results."""
        models = []

        # Handle different JSON structures
        elo_data = data.get("elo_rating", data.get("leaderboard", data))

        if isinstance(elo_data, dict):
            # Format: {model_name: elo_score}
            sorted_models = sorted(elo_data.items(), key=lambda x: x[1], reverse=True)
            for rank, (name, elo) in enumerate(sorted_models[:MAX_LEADERBOARD_SIZE], 1):
                models.append(build_model_dict(
                    name=name,
                    rank=rank,
                    category="llm_api",
                    is_open_source=is_open_source(name),
                    metrics={
                        "elo": elo,
                        "notes": f"Chatbot Arena Elo: {elo}",
                        "source": "lmarena"
                    }
                ))
        elif isinstance(elo_data, list):
            # Format: [{model: name, elo: score}, ...]
            for rank, entry in enumerate(elo_data[:MAX_LEADERBOARD_SIZE], 1):
                name = entry.get("model", entry.get("name", "unknown"))
                elo = entry.get("elo", entry.get("rating", 0))
                models.append(build_model_dict(
                    name=name,
                    rank=rank,
                    category="llm_api",
                    is_open_source=is_open_source(name),
                    metrics={
                        "elo": elo,
                        "notes": f"Chatbot Arena Elo: {elo}",
                        "source": "lmarena"
                    }
                ))

        return models

    def _parse_csv(self, csv_text: str) -> list[dict]:
        """Parse CSV leaderboard data."""
        models = []
        lines = csv_text.strip().split("\n")

        if len(lines) < 2:
            return models

        # Skip header
        for rank, line in enumerate(lines[1:MAX_LEADERBOARD_SIZE+1], 1):
            parts = line.split(",")
            if len(parts) >= 2:
                name = parts[0].strip().strip('"')
                try:
                    elo = float(parts[1].strip())
                except ValueError:
                    elo = 0

                models.append(build_model_dict(
                    name=name,
                    rank=rank,
                    category="llm_api",
                    is_open_source=is_open_source(name),
                    metrics={
                        "elo": elo,
                        "notes": f"Chatbot Arena Elo: {elo}",
                        "source": "lmarena"
                    }
                ))

        return models



# Quick test
if __name__ == "__main__":
    fetcher = LMArenaFetcher()

    print("=== Chatbot Arena Elo Rankings ===")
    rankings = fetcher.fetch_elo_rankings()
    for m in rankings[:10]:
        elo = m["metrics"].get("elo", "N/A")
        badge = "" if m["is_open_source"] else " [CLOSED]"
        print(f"  #{m['sota_rank']} {m['name']}{badge}: Elo {elo}")
