"""
HuggingFace API fetcher.

Fetches data from:
- Open LLM Leaderboard (for local LLMs)
- Model hub (for trending models, downloads)
- Spaces (for leaderboard data)

FREE API with rate limits (~100s requests/hour).
"""

import json
import sys
import urllib.request
import urllib.error
from datetime import datetime
from pathlib import Path
from typing import Optional

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
from constants import HTTP_TIMEOUT_SECONDS, DEFAULT_TRENDING_LIMIT
from utils.models import normalize_model_id
from utils.log import get_logger

logger = get_logger("fetchers.huggingface")


class HuggingFaceFetcher:
    """Fetch SOTA data from HuggingFace APIs."""

    # Open LLM Leaderboard data URL (hosted on HF Spaces)
    LEADERBOARD_URL = "https://huggingface.co/spaces/open-llm-leaderboard/open_llm_leaderboard/resolve/main/data/models.json"

    # Alternative: Direct API for model info
    MODEL_API = "https://huggingface.co/api/models"

    def __init__(self, timeout: int = HTTP_TIMEOUT_SECONDS):
        self.timeout = timeout

    def fetch_llm_leaderboard(self) -> list[dict]:
        """
        Fetch Open LLM Leaderboard data.

        Returns list of models with benchmark scores.
        """
        try:
            # Try the leaderboard space data
            req = urllib.request.Request(
                self.LEADERBOARD_URL,
                headers={"User-Agent": "sota-tracker-mcp/1.0"}
            )
            with urllib.request.urlopen(req, timeout=self.timeout) as resp:
                data = json.loads(resp.read().decode())
                return self._parse_leaderboard(data)
        except urllib.error.HTTPError as e:
            logger.warning(f"HuggingFace leaderboard fetch failed: {e.code}")
            return []
        except Exception as e:
            logger.warning(f"HuggingFace leaderboard error: {e}")
            return []

    def fetch_trending_models(self, task: str = None, limit: int = DEFAULT_TRENDING_LIMIT) -> list[dict]:
        """
        Fetch top models from HuggingFace Hub by downloads.

        Args:
            task: Filter by task (text-generation, image-to-image, etc.)
            limit: Max models to return
        """
        try:
            url = f"{self.MODEL_API}?sort=downloads&limit={limit}"
            if task:
                url += f"&pipeline_tag={task}"

            req = urllib.request.Request(
                url,
                headers={"User-Agent": "sota-tracker-mcp/1.0"}
            )
            with urllib.request.urlopen(req, timeout=self.timeout) as resp:
                models = json.loads(resp.read().decode())
                return self._parse_hub_models(models)
        except Exception as e:
            logger.warning(f"HuggingFace trending error: {e}")
            return []

    def fetch_model_info(self, model_id: str) -> Optional[dict]:
        """Fetch info for a specific model."""
        try:
            url = f"{self.MODEL_API}/{model_id}"
            req = urllib.request.Request(
                url,
                headers={"User-Agent": "sota-tracker-mcp/1.0"}
            )
            with urllib.request.urlopen(req, timeout=self.timeout) as resp:
                return json.loads(resp.read().decode())
        except Exception as e:
            logger.warning(f"HuggingFace model info error for {model_id}: {e}")
            return None

    def _parse_leaderboard(self, data: list) -> list[dict]:
        """Parse Open LLM Leaderboard format into our model format."""
        models = []
        for entry in data[:20]:  # Top 20
            name = entry.get("model_name", "")
            model = {
                "id": normalize_model_id(name),
                "name": name,
                "category": "llm_local",
                "is_open_source": True,
                "release_date": entry.get("submission_date", datetime.now().strftime("%Y-%m-%d")),
                "metrics": {
                    "notes": f"MMLU: {entry.get('MMLU', 'N/A')}, ARC: {entry.get('ARC', 'N/A')}",
                    "mmlu": entry.get("MMLU"),
                    "arc": entry.get("ARC"),
                    "hellaswag": entry.get("HellaSwag"),
                    "truthfulqa": entry.get("TruthfulQA"),
                    "average": entry.get("Average"),
                    "source": "huggingface_leaderboard"
                }
            }
            models.append(model)
        return models

    def _parse_hub_models(self, models: list) -> list[dict]:
        """Parse HuggingFace Hub API response."""
        parsed = []
        for m in models:
            name = m.get("id", "")
            parsed.append({
                "id": normalize_model_id(name),
                "name": name,
                "downloads": m.get("downloads", 0),
                "likes": m.get("likes", 0),
                "pipeline_tag": m.get("pipeline_tag"),
                "last_modified": m.get("lastModified"),
            })
        return parsed


# Quick test
if __name__ == "__main__":
    fetcher = HuggingFaceFetcher()

    print("=== Trending text-generation models ===")
    trending = fetcher.fetch_trending_models(task="text-generation", limit=5)
    for m in trending:
        print(f"  {m['name']}: {m['downloads']} downloads")
