"""
Smart cache manager for SOTA data.

Logic:
- First query of the day for a category → fetch fresh data from sources
- Subsequent queries that day → return cached data (instant)
- Failed fetches → fall back to cached data, log error

This ensures:
- Data is always fresh (max 24h stale)
- Most queries are instant (cached)
- No unnecessary API calls for unused categories
"""

import json
import sqlite3
import threading
from contextlib import contextmanager
from datetime import datetime, date
from pathlib import Path
from typing import Optional

# Import fetchers
from .huggingface import HuggingFaceFetcher
from .lmarena import LMArenaFetcher
from .artificial_analysis import ArtificialAnalysisFetcher


class CacheManager:
    """Manages smart caching of SOTA data."""

    # Map categories to their data sources
    CATEGORY_SOURCES = {
        "llm_local": ["huggingface"],
        "llm_api": ["lmarena", "artificial_analysis"],
        "llm_coding": ["huggingface", "lmarena"],
        "image_gen": ["artificial_analysis"],
        "image_edit": ["artificial_analysis"],
        "video": ["artificial_analysis"],
        "tts": ["artificial_analysis"],
        "stt": ["huggingface"],
        "music": [],  # No good API source
        "3d": [],  # No good API source
        "embeddings": ["huggingface"],
    }

    def __init__(self, db_path: Path):
        self.db_path = db_path
        self.hf_fetcher = HuggingFaceFetcher()
        self.lm_fetcher = LMArenaFetcher()
        self.aa_fetcher = ArtificialAnalysisFetcher()
        # Thread lock to prevent race conditions during cache refresh
        self._refresh_lock = threading.Lock()
        # Per-category locks for finer-grained concurrency
        self._category_locks: dict[str, threading.Lock] = {}

    def get_db(self) -> sqlite3.Connection:
        """
        Get database connection. Caller must close.

        DEPRECATED: Use get_db_context() instead to ensure connections are always closed.
        This method remains for backward compatibility but should not be used in new code.
        """
        db = sqlite3.connect(str(self.db_path), timeout=30.0)
        db.row_factory = sqlite3.Row
        # Enable WAL mode for better concurrent access
        db.execute("PRAGMA journal_mode=WAL")
        return db

    @contextmanager
    def get_db_context(self):
        """Get database connection as context manager (auto-closes)."""
        db = sqlite3.connect(str(self.db_path), timeout=30.0)
        db.row_factory = sqlite3.Row
        # Enable WAL mode for better concurrent access
        db.execute("PRAGMA journal_mode=WAL")
        try:
            yield db
        finally:
            db.close()

    def _get_category_lock(self, category: str) -> threading.Lock:
        """Get or create a lock for a specific category."""
        with self._refresh_lock:
            if category not in self._category_locks:
                self._category_locks[category] = threading.Lock()
            return self._category_locks[category]

    def is_cache_fresh(self, category: str) -> bool:
        """Check if category was fetched today."""
        with self.get_db_context() as db:
            row = db.execute(
                "SELECT last_fetched FROM cache_status WHERE category = ?",
                (category,)
            ).fetchone()

        if not row or not row["last_fetched"]:
            return False

        # Parse timestamp and check if it's today
        try:
            last_fetched = datetime.fromisoformat(row["last_fetched"])
            return last_fetched.date() == date.today()
        except (ValueError, TypeError):
            return False

    def refresh_if_stale(self, category: str) -> dict:
        """
        Refresh category data if stale (not fetched today).

        Thread-safe: Uses per-category locks to prevent race conditions.

        Returns:
            {
                "refreshed": bool,
                "source": str or None,
                "models_updated": int,
                "error": str or None
            }
        """
        # Acquire per-category lock to prevent concurrent refresh attempts
        lock = self._get_category_lock(category)
        with lock:
            # Double-check cache freshness after acquiring lock
            # (another thread may have refreshed while we waited)
            if self.is_cache_fresh(category):
                return {
                    "refreshed": False,
                    "source": "cache",
                    "models_updated": 0,
                    "error": None
                }

            # Need to refresh - fetch from sources
            sources = self.CATEGORY_SOURCES.get(category, [])
            if not sources:
                # No automatic sources for this category, use manual data
                self._update_cache_status(category, "manual", True, None)
                return {
                    "refreshed": True,
                    "source": "manual",
                    "models_updated": 0,
                    "error": None
                }

            # Try each source
            models = []
            source_used = None
            error = None

            for source in sources:
                try:
                    fetched = self._fetch_from_source(source, category)
                    if fetched:
                        models.extend(fetched)
                        source_used = source
                        break  # Got data, stop trying
                except Exception as e:
                    error = str(e)
                    continue

            if models:
                # Update database with fresh data
                count = self._update_models(category, models)
                self._update_cache_status(category, source_used, True, None)
                return {
                    "refreshed": True,
                    "source": source_used,
                    "models_updated": count,
                    "error": None
                }
            else:
                # All sources failed - mark as attempted but keep old data
                self._update_cache_status(category, source_used, False, error)
                return {
                    "refreshed": False,
                    "source": source_used,
                    "models_updated": 0,
                    "error": error or "No data returned from sources"
                }

    def _fetch_from_source(self, source: str, category: str) -> list[dict]:
        """Fetch data from a specific source."""
        if source == "huggingface":
            if category in ["llm_local", "llm_coding"]:
                return self.hf_fetcher.fetch_llm_leaderboard()
            elif category == "embeddings":
                return self.hf_fetcher.fetch_trending_models(task="feature-extraction", limit=10)
            elif category == "stt":
                return self.hf_fetcher.fetch_trending_models(task="automatic-speech-recognition", limit=10)
            else:
                return []

        elif source == "lmarena":
            return self.lm_fetcher.fetch_elo_rankings()

        elif source == "artificial_analysis":
            aa_category_map = {
                "llm_api": "llm",
                "image_gen": "image_gen",
                "image_edit": "image_gen",
                "video": "video",
                "tts": "tts",
            }
            aa_cat = aa_category_map.get(category)
            if aa_cat:
                return self.aa_fetcher.fetch_category(aa_cat)
            return []

        return []

    def _update_models(self, category: str, models: list[dict]) -> int:
        """Update models in database from fresh data using INSERT OR REPLACE."""
        if not models:
            return 0

        count = 0
        now = datetime.now().isoformat()
        with self.get_db_context() as db:
            for model in models:
                # Use INSERT OR REPLACE to upsert in a single query
                # This eliminates the N+1 query pattern (SELECT then INSERT/UPDATE)
                db.execute("""
                    INSERT OR REPLACE INTO models
                    (id, name, category, is_open_source, is_sota, sota_rank, metrics, last_updated, source)
                    VALUES (?, ?, ?, ?, 1, ?, ?, ?, 'auto')
                """, (
                    model["id"],
                    model["name"],
                    model.get("category", category),
                    model.get("is_open_source", True),
                    model.get("sota_rank"),
                    json.dumps(model.get("metrics", {})),
                    now
                ))
                count += 1

            db.commit()
        return count

    def _update_cache_status(self, category: str, source: str, success: bool, error: Optional[str]):
        """Update cache status for a category."""
        with self.get_db_context() as db:
            db.execute("""
                INSERT OR REPLACE INTO cache_status (category, last_fetched, fetch_source, fetch_success, error_message)
                VALUES (?, ?, ?, ?, ?)
            """, (
                category,
                datetime.now().isoformat(),
                source,
                success,
                error
            ))
            db.commit()

    def get_cache_status(self) -> list[dict]:
        """Get cache status for all categories."""
        with self.get_db_context() as db:
            rows = db.execute("SELECT * FROM cache_status ORDER BY category").fetchall()
            return [dict(row) for row in rows]

    def force_refresh(self, category: str) -> dict:
        """Force refresh a category regardless of cache status."""
        # Clear cache status first
        with self.get_db_context() as db:
            db.execute("DELETE FROM cache_status WHERE category = ?", (category,))
            db.commit()

        # Now refresh
        return self.refresh_if_stale(category)


# Singleton instance (created when server starts)
_cache_manager: Optional[CacheManager] = None
_cache_manager_lock = threading.Lock()


def get_cache_manager(db_path: Path) -> CacheManager:
    """Get or create cache manager singleton (thread-safe)."""
    global _cache_manager
    if _cache_manager is None:
        with _cache_manager_lock:
            # Double-checked locking pattern
            if _cache_manager is None:
                _cache_manager = CacheManager(db_path)
    return _cache_manager
