# Security Policy

## Supported Versions

| Version | Supported Until |
|---------|------------------|
| Current | Ongoing |

## Reporting a Vulnerability

### Security Best Practices

This project follows these security practices:

1. **No Secrets in Code**: No API keys, tokens, or passwords in the repository
2. **User-Specific Data Ignored**: `data/hardware_profiles.json` is in `.gitignore`
3. **Read-Only Scraping**: Scrapers only fetch public data, no authentication required
4. **Rate-Limited Scraping**: Minimum 1 second delay between requests
5. **CORS Limited**: REST API is read-only with wildcard CORS

### Vulnerability Disclosure

If you discover a security vulnerability, please:

1. **Do NOT**: Create a public issue
2. **Do**: Send a private message or email to the maintainers
3. **Include**: Description of the vulnerability, steps to reproduce, and suggested fix
4. **Response**: We will acknowledge within 48 hours and fix within 7 days

### Common Vulnerability Types

Here are areas to be aware of:

#### Dependency Vulnerabilities

We scan dependencies regularly:

```bash
pip-audit  # Check for known vulnerabilities
```

Report any CVEs found in `requirements.txt`.

#### Data Integrity

- Database is read-only for public API
- All scrapers validate data before insertion
- SQL injection prevented via parameterized queries

#### Rate Limiting

The REST API has built-in protections:

- 60 requests/minute for standard endpoints
- 10 requests/minute for expensive operations (compare, hardware)
- IP-based rate limiting via `slowapi`

## Securing Your Installation

If you run your own instance:

### 1. Environment Variables

Optional environment variables (not required for basic use):

```bash
# Set these before running
export SOTA_CACHE_DIR=/path/to/cache
export SOTA_LOG_LEVEL=INFO
```

### 2. Reverse Proxy

For production deployments, use nginx or Caddy with HTTPS:

```nginx
location / {
    proxy_pass http://127.0.0.1:8000;
    proxy_set_header X-Real-IP $remote_addr;
}
```

### 3. Firewall

Restrict access if needed:

```bash
# Allow localhost and trusted IP
sudo ufw allow from 192.168.1.0/24 to any port 8000
```

### 4. Authentication (Optional)

If you need authentication, modify `rest_api.py`:

```python
from fastapi import Depends, HTTPException, status
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials

security = HTTPBearer()

async def verify_token(credentials: HTTPAuthorizationCredentials = Depends(security)):
    # Add your token verification logic
    return True

@app.get("/api/v1/models", dependencies=[Depends(verify_token)])
def list_models(...)
```

## Private Mode Usage

If you want to keep your data private:

### Fork and Run Locally

```bash
# Fork this repo
# Disable GitHub Actions in your fork settings
# Run scrapers locally
python scrapers/run_all.py

# Data stays on your machine
```

### Cloud Deployment

For VPS deployment:

1. Use a trusted cloud provider
2. Enable HTTPS (Let's Encrypt)
3. Configure firewall rules
4. Monitor logs for suspicious activity

## Data Sources and Attribution

We only scrape from sources that:

- Allow scraping in `robots.txt` or explicitly welcome it (e.g., "Open to AI crawlers")
- Provide public APIs or publicly accessible data
- Don't require authentication

Current data sources and their scraping policies:

| Source | Policy | Notes |
|--------|--------|-------|
| LMArena | `robots.txt: Allow: /` | Public API |
| Artificial Analysis | Explicitly allows AI crawlers | Public data |
| HuggingFace | Public API | Official API |
| Civitai | Public API | Model metadata |

If a source changes policy, we immediately stop scraping that source.

## Logging

Default logging outputs to stdout. For production:

```python
import logging

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('/var/log/sota-tracker.log'),
        logging.StreamHandler()
    ]
)
```

### Sensitive Information

Logs never include:
- User data (hardware_profiles.json is logged as "Profile configured")
- Passwords or tokens
- Personal information

## Compliance

### GDPR / Data Privacy

This project:

- Collects no personal data
- Only stores benchmark rankings (fact, not personal data)
- Does not track users
- Provides download for personal data (the JSON export)

### Terms of Service

All data is aggregated from sources that govern their own terms of use. We:

- Attribute all sources
- Don't claim ownership of rankings
- Provide fair use aggregation
- Don't compete commercially with sources

## Security Audits

We welcome security audits:

1. Create an issue marked `[Security Audit]`
2. We will provide guidance on scope
3. Report findings privately
4. Acknowledge contributions (if desired)

## Contact

For security issues only:
- Open a private issue or
- Contact via repository security email (if configured)

For non-security issues, use standard issues/discussions.