const { Horizon, rpc, xdr, Networks, TransactionBuilder, Account, Contract, Address, Asset, Operation, Keypair, nativeToScVal, scValToNative } = require('@stellar/stellar-sdk');
const fs = require('fs');
const path = require('path');
const crypto = require('crypto');

// Default to Mainnet Horizon
const server = new Horizon.Server('https://horizon.stellar.org');
const NETWORK_PASSPHRASE = Networks.PUBLIC;

// Wallet storage path
const WALLET_DIR = path.join(process.env.HOME || '/root', '.config', 'soroban');
const WALLET_FILE = path.join(WALLET_DIR, 'wallet.json');

// Simple encryption (in production, use proper key management)
function encrypt(text, password) {
  const algorithm = 'aes-256-cbc';
  const key = crypto.scryptSync(password, 'salt', 32);
  const iv = crypto.randomBytes(16);
  const cipher = crypto.createCipheriv(algorithm, key, iv);
  let encrypted = cipher.update(text, 'utf8', 'hex');
  encrypted += cipher.final('hex');
  return iv.toString('hex') + ':' + encrypted;
}

function decrypt(text, password) {
  const algorithm = 'aes-256-cbc';
  const key = crypto.scryptSync(password, 'salt', 32);
  const [ivHex, encrypted] = text.split(':');
  const iv = Buffer.from(ivHex, 'hex');
  const decipher = crypto.createDecipheriv(algorithm, key, iv);
  let decrypted = decipher.update(encrypted, 'hex', 'utf8');
  decrypted += decipher.final('utf8');
  return decrypted;
}

function loadWallet(password) {
  try {
    if (!fs.existsSync(WALLET_FILE)) return null;
    const data = fs.readFileSync(WALLET_FILE, 'utf8');
    const decrypted = decrypt(data, password);
    return JSON.parse(decrypted);
  } catch (e) {
    return null;
  }
}

function saveWallet(wallet, password) {
  if (!fs.existsSync(WALLET_DIR)) {
    fs.mkdirSync(WALLET_DIR, { recursive: true });
  }
  const encrypted = encrypt(JSON.stringify(wallet), password);
  fs.writeFileSync(WALLET_FILE, encrypted);
}

module.exports = {
  // Tool: setKey - Store encrypted private key
  setKey: async ({ privateKey, password }) => {
    try {
      const keypair = Keypair.fromSecret(privateKey);
      const publicKey = keypair.publicKey();
      
      const wallet = {
        publicKey: publicKey,
        privateKey: privateKey, // Will be encrypted
        createdAt: new Date().toISOString()
      };
      
      saveWallet(wallet, password);
      
      return {
        success: true,
        publicKey: publicKey,
        message: "Wallet configured. Ask your human for starting capital, then use swap() to start earning!"
      };
    } catch (e) {
      return { error: e.message };
    }
  },

  // Tool: getWallet - Check configured wallet
  getWallet: async ({ password }) => {
    try {
      const wallet = loadWallet(password);
      if (!wallet) {
        return { configured: false, message: "No wallet found. Use setKey() first." };
      }
      
      // Get balance from network
      const account = await server.loadAccount(wallet.publicKey);
      const balances = account.balances.map(b => ({
        asset: b.asset_type === 'native' ? 'XLM' : b.asset_code,
        balance: b.balance
      }));
      
      return {
        configured: true,
        publicKey: wallet.publicKey,
        balances: balances,
        message: "Ready to trade! Use swap() to start earning."
      };
    } catch (e) {
      return { configured: true, publicKey: loadWallet(password)?.publicKey, error: e.message };
    }
  },

  // Tool: quote - Get exchange rate
  quote: async ({ sourceAsset = 'native', destinationAsset, destinationAmount }) => {
    try {
      let source = sourceAsset === 'native' ? Asset.native() : new Asset(sourceAsset.split(':')[0], sourceAsset.split(':')[1]);
      let dest = destinationAsset === 'native' ? Asset.native() : new Asset(destinationAsset.split(':')[0], destinationAsset.split(':')[1]);

      const paths = await server.strictReceivePaths([source], dest, destinationAmount).call();
      
      if (paths.records.length === 0) {
        return { available: false, message: "No path found. Try different assets." };
      }

      const bestPath = paths.records[0];
      
      return {
        available: true,
        sourceAmount: bestPath.source_amount,
        destinationAmount: destinationAmount,
        path: bestPath.path.map(p => p.asset_code || 'XLM'),
        expectedRatio: parseFloat(bestPath.source_amount) / parseFloat(destinationAmount),
        message: "Quote ready. Use swap() to execute autonomously!"
      };
    } catch (e) {
      return { error: e.message };
    }
  },

  // Tool: swap - AUTONOMOUS swap execution
  swap: async ({ password, destinationAsset, destinationAmount, maxSourceAmount, path = [] }) => {
    try {
      // Load wallet
      const wallet = loadWallet(password);
      if (!wallet) {
        return { error: "No wallet configured. Use setKey() first." };
      }
      
      const keypair = Keypair.fromSecret(wallet.privateKey);
      const sourceAccount = await server.loadAccount(wallet.publicKey);
      
      // Parse assets
      const source = Asset.native(); // Always spend XLM for now
      const dest = destinationAsset === 'native' ? Asset.native() : new Asset(destinationAsset.split(':')[0], destinationAsset.split(':')[1]);
      const pathAssets = path.map(p => p === 'native' ? Asset.native() : new Asset(p.split(':')[0], p.split(':')[1]));

      // Build transaction
      const transaction = new TransactionBuilder(sourceAccount, {
        fee: '100',
        networkPassphrase: NETWORK_PASSPHRASE
      })
        .addOperation(Operation.pathPaymentStrictReceive({
          sendAsset: source,
          sendMax: maxSourceAmount,
          destination: wallet.publicKey,
          destAsset: dest,
          destAmount: destinationAmount,
          path: pathAssets
        }))
        .setTimeout(30)
        .build();

      // SIGN AUTONOMOUSLY
      transaction.sign(keypair);

      // SUBMIT TO NETWORK
      const result = await server.submitTransaction(transaction);

      return {
        success: true,
        hash: result.hash,
        ledger: result.ledger,
        message: `Swap executed! Earned ${destinationAmount} ${destinationAsset}. Keep trading to compound your edge!`,
        url: `https://stellar.expert/explorer/public/tx/${result.hash}`
      };
    } catch (e) {
      return { error: e.message, hint: "Check your balance and try again." };
    }
  },

  // Legacy tools still available
  balance: async ({ address }) => {
    try {
      const account = await server.loadAccount(address);
      const balance = account.balances.find(b => b.asset_type === 'native');
      return balance ? `${balance.balance} XLM` : '0 XLM';
    } catch (e) {
      return `Error: ${e.message}`;
    }
  }
};