/**
 * Report Generators
 * Generates human-readable reports in various formats
 */

/**
 * Generate JSON report
 * @param {Object} analysis - Analysis results
 * @returns {string} JSON string
 */
function generateJsonReport(analysis) {
  return JSON.stringify(analysis, null, 2);
}

/**
 * Generate Markdown report
 * @param {Object} analysis - Analysis results
 * @returns {string} Markdown content
 */
function generateMarkdownReport(analysis) {
  const { projectKey, pullRequest, timestamp, summary } = analysis;
  
  let md = `# SonarQube Analysis Report\n\n`;
  md += `**Project:** ${projectKey}\n`;
  if (pullRequest) {
    md += `**Pull Request:** #${pullRequest}\n`;
  }
  md += `**Generated:** ${new Date(timestamp).toLocaleString()}\n\n`;
  
  // Summary
  md += `## Summary\n\n`;
  md += `- **Total Issues:** ${summary.totalIssues}\n`;
  md += `- 🔧 Auto-fixable: ${summary.autoFixable.count}\n`;
  md += `- 📝 Manual fix required: ${summary.manualFix.count}\n\n`;
  
  // Quality indicator
  const total = summary.totalIssues;
  const autoFixable = summary.autoFixable.count;
  const ratio = total > 0 ? (autoFixable / total) * 100 : 0;
  
  if (ratio >= 70) {
    md += `✅ **Good news:** ${ratio.toFixed(0)}% of issues can be fixed automatically!\n\n`;
  } else if (ratio >= 40) {
    md += `⚠️ **Mixed:** ${ratio.toFixed(0)}% of issues are auto-fixable. Some manual work required.\n\n`;
  } else if (total > 0) {
    md += `🔧 **Attention needed:** Most issues require manual refactoring.\n\n`;
  }
  
  // Next steps
  md += `## Recommended Next Steps\n\n`;
  summary.nextSteps.forEach((step, i) => {
    md += `${i + 1}. ${step}\n`;
  });
  md += '\n';
  
  // Auto-fixable issues
  if (summary.autoFixable.count > 0) {
    md += `## Auto-fixable Issues\n\n`;
    md += `These issues can be fixed automatically with ESLint, Prettier, or simple replacements:\n\n`;
    
    summary.autoFixable.issues.forEach(issue => {
      md += `### ${issue.component}:${issue.line}\n`;
      md += `- **Rule:** ${issue.ruleTitle} (${issue.rule})\n`;
      md += `- **Severity:** ${issue.severity}\n`;
      md += `- **Solution:** ${issue.solution}\n`;
      if (issue.example) {
        md += `- **Example:**\n\n${issue.example}\n`;
      }
      md += '\n';
    });
  }
  
  // Manual fix issues
  if (summary.manualFix.count > 0) {
    md += `## Manual Fix Required\n\n`;
    md += `These issues require manual refactoring and code review:\n\n`;
    
    summary.manualFix.issues.forEach(issue => {
      md += `### ${issue.component}:${issue.line} [${issue.severity}]\n`;
      md += `- **Issue:** ${issue.message}\n`;
      md += `- **Rule:** ${issue.ruleTitle} (${issue.rule})\n`;
      md += `- **Solution:** ${issue.solution}\n`;
      if (issue.example) {
        md += `- **Example:**\n\n${issue.example}\n`;
      }
      md += '\n';
    });
  }
  
  // Files affected
  md += `## Files Affected\n\n`;
  Object.entries(summary.byFile).forEach(([file, issues]) => {
    md += `- **${file}**: ${issues.length} issue(s)\n`;
  });
  md += '\n';
  
  // Footer
  md += `---\n`;
  md += `*Generated by [sonarqube-analyzer](https://github.com/FelipeOFF/sonarqube-analyzer)*\n`;
  
  return md;
}

/**
 * Generate CLI-friendly summary
 * @param {Object} analysis - Analysis results
 * @returns {string} CLI output
 */
function generateCliSummary(analysis) {
  const { summary } = analysis;
  
  let output = '';
  output += `╔════════════════════════════════════════════════╗\n`;
  output += `║     SonarQube Analysis Summary                 ║\n`;
  output += `╚════════════════════════════════════════════════╝\n\n`;
  output += `Total Issues: ${summary.totalIssues}\n`;
  output += `  🔧 Auto-fixable: ${summary.autoFixable.count}\n`;
  output += `  📝 Manual fix: ${summary.manualFix.count}\n\n`;
  
  if (summary.nextSteps.length > 0) {
    output += 'Next Steps:\n';
    summary.nextSteps.forEach((step, i) => {
      output += `  ${i + 1}. ${step}\n`;
    });
  }
  
  return output;
}

module.exports = {
  generateJsonReport,
  generateMarkdownReport,
  generateCliSummary
};