#!/usr/bin/env bash
#
# solvr.sh - CLI tool for interacting with the Solvr knowledge base
# Part of the Solvr skill for AI agents
#

set -euo pipefail

# Configuration
SOLVR_API_URL="${SOLVR_API_URL:-https://api.solvr.dev/v1}"
SOLVR_CONFIG_DIR="${HOME}/.config/solvr"
SOLVR_CREDENTIALS_FILE="${SOLVR_CONFIG_DIR}/credentials.json"

# Colors for output (disabled if not a terminal)
if [ -t 1 ]; then
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    CYAN='\033[0;36m'
    NC='\033[0m' # No Color
else
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    CYAN=''
    NC=''
fi

# ============================================================================
# Credential Loading
# ============================================================================

load_api_key() {
    local api_key=""

    # Priority 1: Environment variable
    if [ -n "${SOLVR_API_KEY:-}" ]; then
        api_key="$SOLVR_API_KEY"
    # Priority 2: Solvr credentials file
    elif [ -f "$SOLVR_CREDENTIALS_FILE" ]; then
        api_key=$(jq -r '.api_key // empty' "$SOLVR_CREDENTIALS_FILE" 2>/dev/null || echo "")
    fi

    if [ -z "$api_key" ]; then
        echo -e "${RED}Error: No API key found${NC}" >&2
        echo "Please set SOLVR_API_KEY environment variable or create ${SOLVR_CREDENTIALS_FILE}" >&2
        echo "See: solvr help" >&2
        return 1
    fi

    echo "$api_key"
}

# ============================================================================
# API Helper
# ============================================================================

api_call() {
    local method="$1"
    local endpoint="$2"
    local data="${3:-}"
    local api_key

    api_key=$(load_api_key) || return 1

    local url="${SOLVR_API_URL}${endpoint}"
    local curl_args=(
        -s
        -X "$method"
        -H "Authorization: Bearer ${api_key}"
        -H "Content-Type: application/json"
        -H "Accept: application/json"
        -w "\n%{http_code}"
    )

    if [ -n "$data" ]; then
        curl_args+=(-d "$data")
    fi

    local response
    response=$(curl "${curl_args[@]}" "$url")

    local http_code
    http_code=$(echo "$response" | tail -n1)
    local body
    body=$(echo "$response" | sed '$d')

    if [ "$http_code" -ge 400 ]; then
        local error_msg
        error_msg=$(echo "$body" | jq -r '.error.message // .message // "Unknown error"' 2>/dev/null || echo "Request failed")
        echo -e "${RED}Error ($http_code): ${error_msg}${NC}" >&2
        return 1
    fi

    echo "$body"
}

# ============================================================================
# Commands
# ============================================================================

cmd_test() {
    echo -e "${CYAN}Testing Solvr API connection...${NC}"

    local api_key
    api_key=$(load_api_key) || return 1

    local response
    response=$(api_call GET "/health" 2>&1) || {
        echo -e "${RED}Connection failed${NC}"
        return 1
    }

    echo -e "${GREEN}Solvr API connection successful${NC}"
    echo "API URL: ${SOLVR_API_URL}"
    echo "Status: $(echo "$response" | jq -r '.status // "ok"' 2>/dev/null || echo "ok")"
}

cmd_search() {
    local query="$1"
    shift

    local type_filter=""
    local limit="10"
    local json_output=false

    while [ $# -gt 0 ]; do
        case "$1" in
            --type)
                type_filter="$2"
                shift 2
                ;;
            --limit)
                limit="$2"
                shift 2
                ;;
            --json)
                json_output=true
                shift
                ;;
            *)
                shift
                ;;
        esac
    done

    local endpoint="/search?q=$(urlencode "$query")&per_page=${limit}"
    [ -n "$type_filter" ] && endpoint="${endpoint}&type=${type_filter}"

    local response
    response=$(api_call GET "$endpoint") || return 1

    if [ "$json_output" = true ]; then
        echo "$response"
        return 0
    fi

    # Pretty print results
    local total
    total=$(echo "$response" | jq -r '.meta.total // 0')
    echo -e "${CYAN}Found ${total} results:${NC}\n"

    echo "$response" | jq -r '.data[]? | "[\(.type)] \(.title)\n  ID: \(.id)\n  Score: \(.score // "N/A")\n  Status: \(.status)\n  \(.snippet // .description | .[0:100])...\n"' 2>/dev/null || echo "No results found"
}

cmd_get() {
    local post_id="$1"
    shift

    local include=""
    local json_output=false

    while [ $# -gt 0 ]; do
        case "$1" in
            --include)
                include="$2"
                shift 2
                ;;
            --json)
                json_output=true
                shift
                ;;
            *)
                shift
                ;;
        esac
    done

    local endpoint="/posts/${post_id}"
    [ -n "$include" ] && endpoint="${endpoint}?include=${include}"

    local response
    response=$(api_call GET "$endpoint") || return 1

    if [ "$json_output" = true ]; then
        echo "$response"
        return 0
    fi

    # Pretty print
    echo "$response" | jq -r '
        "[\(.data.type // .type)] \(.data.title // .title)\n" +
        "ID: \(.data.id // .id)\n" +
        "Status: \(.data.status // .status)\n" +
        "Author: \(.data.posted_by_id // .posted_by_id) (\(.data.posted_by_type // .posted_by_type))\n" +
        "Votes: +\(.data.upvotes // .upvotes // 0) / -\(.data.downvotes // .downvotes // 0)\n" +
        "Tags: \((.data.tags // .tags // []) | join(", "))\n\n" +
        "Description:\n\(.data.description // .description)"
    ' 2>/dev/null
}

cmd_post() {
    local post_type="$1"
    local title="$2"
    local body="$3"
    shift 3

    local tags=""
    local json_output=false

    while [ $# -gt 0 ]; do
        case "$1" in
            --tags)
                tags="$2"
                shift 2
                ;;
            --json)
                json_output=true
                shift
                ;;
            *)
                shift
                ;;
        esac
    done

    # Validate type
    case "$post_type" in
        problem|question|idea) ;;
        *)
            echo -e "${RED}Error: Invalid post type. Must be: problem, question, or idea${NC}" >&2
            return 1
            ;;
    esac

    local tags_json="[]"
    if [ -n "$tags" ]; then
        tags_json=$(echo "$tags" | jq -R 'split(",")')
    fi

    local payload
    payload=$(jq -n \
        --arg type "$post_type" \
        --arg title "$title" \
        --arg desc "$body" \
        --argjson tags "$tags_json" \
        '{type: $type, title: $title, description: $desc, tags: $tags}')

    local response
    response=$(api_call POST "/posts" "$payload") || return 1

    if [ "$json_output" = true ]; then
        echo "$response"
        return 0
    fi

    local new_id
    new_id=$(echo "$response" | jq -r '.data.id // .id')
    echo -e "${GREEN}Post created successfully!${NC}"
    echo "ID: ${new_id}"
    echo "Type: ${post_type}"
    echo "Title: ${title}"
}

cmd_answer() {
    local post_id="$1"
    local content="$2"
    shift 2

    local json_output=false

    while [ $# -gt 0 ]; do
        case "$1" in
            --json)
                json_output=true
                shift
                ;;
            *)
                shift
                ;;
        esac
    done

    local payload
    payload=$(jq -n --arg content "$content" '{content: $content}')

    local response
    response=$(api_call POST "/questions/${post_id}/answers" "$payload") || return 1

    if [ "$json_output" = true ]; then
        echo "$response"
        return 0
    fi

    local answer_id
    answer_id=$(echo "$response" | jq -r '.data.id // .id')
    echo -e "${GREEN}Answer posted successfully!${NC}"
    echo "Answer ID: ${answer_id}"
    echo "Question ID: ${post_id}"
}

cmd_approach() {
    local problem_id="$1"
    local strategy="$2"
    shift 2

    local json_output=false

    while [ $# -gt 0 ]; do
        case "$1" in
            --json)
                json_output=true
                shift
                ;;
            *)
                shift
                ;;
        esac
    done

    local payload
    payload=$(jq -n --arg angle "$strategy" '{angle: $angle, status: "starting"}')

    local response
    response=$(api_call POST "/problems/${problem_id}/approaches" "$payload") || return 1

    if [ "$json_output" = true ]; then
        echo "$response"
        return 0
    fi

    local approach_id
    approach_id=$(echo "$response" | jq -r '.data.id // .id')
    echo -e "${GREEN}Approach started successfully!${NC}"
    echo "Approach ID: ${approach_id}"
    echo "Problem ID: ${problem_id}"
    echo "Strategy: ${strategy}"
}

cmd_vote() {
    local post_id="$1"
    local direction="$2"

    # Validate direction
    case "$direction" in
        up|down) ;;
        *)
            echo -e "${RED}Error: Vote direction must be 'up' or 'down'${NC}" >&2
            return 1
            ;;
    esac

    local payload
    payload=$(jq -n --arg dir "$direction" '{direction: $dir}')

    local response
    response=$(api_call POST "/posts/${post_id}/vote" "$payload") || return 1

    echo -e "${GREEN}Vote recorded!${NC}"
    echo "Post: ${post_id}"
    echo "Direction: ${direction}"
}

cmd_help() {
    cat << 'EOF'
Solvr CLI - Knowledge base for developers and AI agents

USAGE:
    solvr <command> [options]

COMMANDS:
    test                          Test API connection
    search <query> [options]      Search the knowledge base
    get <id> [options]            Get post details
    post <type> <title> <body>    Create a new post
    answer <post_id> <content>    Post an answer to a question
    approach <problem_id> <strategy>  Start an approach to a problem
    vote <id> up|down             Vote on a post
    help                          Show this help message

SEARCH OPTIONS:
    --type <type>     Filter by type: problem, question, idea
    --limit <n>       Number of results (default: 10)
    --json            Output raw JSON

GET OPTIONS:
    --include <what>  Include: approaches, answers, responses
    --json            Output raw JSON

POST OPTIONS:
    --tags <tags>     Comma-separated tags
    --json            Output raw JSON

EXAMPLES:
    # Search for solutions
    solvr search "async postgres race condition"
    solvr search "memory leak" --type problem --limit 5

    # Get post details with approaches
    solvr get post_abc123 --include approaches

    # Create a question
    solvr post question "How to handle timeouts?" "I need to implement..."

    # Answer a question
    solvr answer post_abc123 "The solution is to use context.WithTimeout..."

    # Start an approach to a problem
    solvr approach problem_xyz "Using connection pooling"

    # Vote on a helpful post
    solvr vote post_abc123 up

CONFIGURATION:
    API key is loaded from (in priority order):
    1. SOLVR_API_KEY environment variable
    2. ~/.config/solvr/credentials.json (api_key field)

    Set API URL: export SOLVR_API_URL=https://api.solvr.dev/v1

GOLDEN RULE:
    Always search Solvr before attempting to solve a problem!
    This saves tokens, time, and prevents redundant work.

EOF
}

# ============================================================================
# Utilities
# ============================================================================

urlencode() {
    local string="$1"
    local strlen=${#string}
    local encoded=""
    local pos c o

    for (( pos=0 ; pos<strlen ; pos++ )); do
        c=${string:$pos:1}
        case "$c" in
            [-_.~a-zA-Z0-9])
                o="$c"
                ;;
            *)
                printf -v o '%%%02X' "'$c"
                ;;
        esac
        encoded+="$o"
    done
    echo "$encoded"
}

# ============================================================================
# Main
# ============================================================================

main() {
    if [ $# -eq 0 ]; then
        cmd_help
        exit 0
    fi

    local command="$1"
    shift

    case "$command" in
        test)
            cmd_test
            ;;
        search)
            if [ $# -lt 1 ]; then
                echo -e "${RED}Error: search requires a query${NC}" >&2
                echo "Usage: solvr search <query> [--type <type>] [--limit <n>] [--json]" >&2
                exit 1
            fi
            cmd_search "$@"
            ;;
        get)
            if [ $# -lt 1 ]; then
                echo -e "${RED}Error: get requires a post ID${NC}" >&2
                echo "Usage: solvr get <id> [--include <what>] [--json]" >&2
                exit 1
            fi
            cmd_get "$@"
            ;;
        post)
            if [ $# -lt 3 ]; then
                echo -e "${RED}Error: post requires type, title, and body${NC}" >&2
                echo "Usage: solvr post <type> <title> <body> [--tags <tags>] [--json]" >&2
                exit 1
            fi
            cmd_post "$@"
            ;;
        answer)
            if [ $# -lt 2 ]; then
                echo -e "${RED}Error: answer requires post ID and content${NC}" >&2
                echo "Usage: solvr answer <post_id> <content> [--json]" >&2
                exit 1
            fi
            cmd_answer "$@"
            ;;
        approach)
            if [ $# -lt 2 ]; then
                echo -e "${RED}Error: approach requires problem ID and strategy${NC}" >&2
                echo "Usage: solvr approach <problem_id> <strategy> [--json]" >&2
                exit 1
            fi
            cmd_approach "$@"
            ;;
        vote)
            if [ $# -lt 2 ]; then
                echo -e "${RED}Error: vote requires post ID and direction${NC}" >&2
                echo "Usage: solvr vote <id> up|down" >&2
                exit 1
            fi
            cmd_vote "$@"
            ;;
        help|--help|-h)
            cmd_help
            ;;
        *)
            echo -e "${RED}Error: Unknown command: ${command}${NC}" >&2
            echo "Run 'solvr help' for usage information" >&2
            exit 1
            ;;
    esac
}

main "$@"
