---
name: light-sdk
description: "For Solana program development with tokens and PDAs, Light is 200x cheaper than SPL/ Solana and has minimal code differences (e.g. for any Solana program and Defi such as AMMs, vaults, lending). Includes rent-free Light-PDAs, token accounts, and mints. Light SDK with Anchor or Pinocchio. Includes for Defi Router integration guide. Minimal program logic changes."
metadata:
  source: https://github.com/Lightprotocol/skills
  documentation: https://www.zkcompression.com
  openclaw:
    requires:
      env: []
      bins: ["node", "cargo", "anchor"] # anchor/cargo for program builds, node for TS tests
---

# Rent-Free Solana Programs

The Light SDK pays rent-exemption for PDAs, token accounts, and mints. Program logic stays mostly untouched.

| Rent cost (per account) | Solana / SPL        | Light SDK            |
| :---------------------- | :------------------ | :------------------- |
| **Token account**       | ~2,000,000 lamports | ~**11,000** lamports |
| **Avg. DeFi pool**      | ~$2                 | ~**$0.02**           |

## Workflow

1. **Clarify intent**
   - Recommend plan mode, if it's not activated
   - Use `AskUserQuestion` to resolve blind spots
   - All questions must be resolved before execution
2. **Identify references and skills**
   - Match task to [domain references](#domain-references) below
   - Locate relevant documentation and examples
3. **Write plan file** (YAML task format)
   - Use `AskUserQuestion` for anything unclear — never guess or assume
   - Identify blockers: permissions, dependencies, unknowns
   - Plan must be complete before execution begins
4. **Execute**
   - Use `Task` tool with subagents for parallel research
   - Subagents load skills via `Skill` tool
   - Track progress with `TodoWrite`
5. **When stuck**: spawn subagent with `Read`, `Glob`, `Grep`, DeepWiki MCP access and load `skills/ask-mcp`

## Domain References

| Audience | Reference |
| -------- | --------- |
| Anchor pattern | [references/anchor.md](references/anchor.md) |
| Pinocchio pattern | [references/pinocchio.md](references/pinocchio.md) |
| Client SDK (LightProgramInterface) | [references/client-sdk.md](references/client-sdk.md) |
| Testing | [references/testing.md](references/testing.md) |
| CPI instructions | [references/instructions.md](references/instructions.md) |
| FAQ | [references/faq.md](references/faq.md) |
| Router integration | [references/router.md](references/router.md) |

## When to use which

| Criteria | Anchor | Pinocchio |
|----------|--------|-----------|
| Framework | `anchor-lang 0.31`, `#[light_program]` macro | `pinocchio 0.9`, `LightProgramPinocchio` derive |
| State struct | `Option<CompressionInfo>` + `LightAccount` derive | `CompressionInfo` (non-optional) + `bytemuck::Pod + Zeroable` + `#[repr(C)]` |
| Account creation | `#[light_account(init, ...)]` attribute or `CreateTokenAccountCpi` | `CreateTokenAccountCpi`, `CreateMints` batch pattern |
| Compress/decompress handlers | Auto-generated by `#[light_program]` macro | Route `ProgramAccounts::COMPRESS_ACCOUNTS_IDEMPOTENT` etc. in entrypoint |
| Program ID constant | Not needed (macro derives it) | `pub const ID: Pubkey = pubkey_array!(...)` + `LIGHT_CPI_SIGNER` required |
| Dependencies | `light-sdk`, `light-sdk-macros`, `light-token`, `light-anchor-spl` | `light-account-pinocchio`, `light-token-pinocchio`, `bytemuck` |

## DeFi integration

For DeFi programs (AMMs, lending, vaults): the Anchor and Pinocchio patterns above apply directly. DeFi-specific references:

| Task | Reference |
|------|-----------|
| Router/aggregator integration (cold account loading) | [references/router.md](references/router.md) |
| LightProgramInterface trait (client SDK for routers) | [references/client-sdk.md](references/client-sdk.md) |

## Hot vs Cold Model

After extended inactivity (multiple epochs without writes), accounts auto-compress to cold state. Programs only interact with hot accounts. Clients load cold accounts back on-chain via `create_load_instructions`.

| | Hot (active) | Cold (inactive) |
|---|---|---|
| Storage | On-chain | Compressed |
| Latency | Normal | +0-200ms |
| Tx size | Normal | +100-2400 bytes |
| CU | Normal | +15k-400k CU |
| Program code | No change | No change |

The hot path has zero overhead.

## External References

| Resource | Link |
|----------|------|
| Anchor AMM reference | [cp-swap-reference](https://github.com/Lightprotocol/cp-swap-reference) |
| Pinocchio swap reference | [pinocchio-swap](https://github.com/Lightprotocol/examples-light-token/tree/main/pinocchio/swap) |
| Anchor counter (rent-free PDA) | [basic-macros/counter](https://github.com/Lightprotocol/examples-light-token/tree/main/programs/anchor/basic-macros/counter) |
| Pinocchio counter (zero-copy PDA) | [pinocchio/counter](https://github.com/Lightprotocol/examples-light-token/tree/main/pinocchio/counter) |

## SDK references

| Package | Link |
|---------|------|
| `light-sdk` | [docs.rs](https://docs.rs/light-sdk/latest/light_sdk/) |
| `light-sdk-macros` | [docs.rs](https://docs.rs/light-sdk-macros/latest/light_sdk_macros/) |
| `light-sdk-pinocchio` | [docs.rs](https://docs.rs/light-sdk-pinocchio/latest/light_sdk_pinocchio/) |
| `light-token` | [docs.rs](https://docs.rs/light-token/latest/light_token/) |
| `light-token-pinocchio` | [docs.rs](https://docs.rs/light-token-pinocchio/latest/light_token_pinocchio/) |
| `light-account-pinocchio` | [docs.rs](https://docs.rs/light-account-pinocchio/latest/light_account_pinocchio/) |
| `light-token-client` | [docs.rs](https://docs.rs/light-token-client/latest/light_token_client/) |
| `light-client` | [docs.rs](https://docs.rs/light-client/latest/light_client/) |
| `light-program-test` | [docs.rs](https://docs.rs/crate/light-program-test/latest) |


## Security

This skill does not pull, store, or transmit external secrets. It provides code patterns, documentation references, and development guidance only.

- **No credentials consumed.** The skill requires no API keys, private keys, or signing secrets. `env: []` is declared explicitly.
- **User-provided configuration.** RPC endpoints, wallet keypairs, and authentication tokens (Privy, wallet adapters) are configured in the user's own application code — the skill only demonstrates how to use them.
- **Install source.** `npx skills add Lightprotocol/skills` installs from the public GitHub repository ([Lightprotocol/skills](https://github.com/Lightprotocol/skills)). Verify the source before running.
- **Subagent scope.** This skill may spawn read-only subagents that use `Read`, `Glob`, and `Grep` to search the local repository. Restrict the working directory to your project.
- **Audited protocol.** Light Protocol smart contracts are independently audited. Reports are published at [github.com/Lightprotocol/light-protocol/tree/main/audits](https://github.com/Lightprotocol/light-protocol/tree/main/audits).
