# 自媒体文案生成器 - 技术方案

## 🛠️ 技术栈选型

### 核心框架
| 组件 | 技术选型 | 理由 |
|------|----------|------|
| **运行时** | Node.js 20+ | 与 OpenClaw 生态一致，异步处理优势 |
| **AI 模型** | Dashscope Qwen3.5+ | 中文优化，成本低，响应快 |
| **Skill 框架** | OpenClaw Skills | 原生集成，无需额外部署 |
| **配置管理** | YAML/JSON | 灵活易维护 |

### 依赖库
```json
{
  "prompts": "Handlebars 模板引擎",
  "validation": "Zod 运行时验证",
  "utils": "lodash 工具函数",
  "logging": "winston 日志"
}
```

---

## 🏗️ 架构设计

### 整体架构
```
┌─────────────────────────────────────────────────────────┐
│                    用户接口层                             │
│  (OpenClaw Message / CLI / Web UI)                      │
└────────────────────┬────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────┐
│                    Skill 入口层                           │
│  - 命令解析 (copywriter generate/optimize/tags)          │
│  - 参数验证                                              │
│  - 上下文管理                                            │
└────────────────────┬────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────┐
│                    业务逻辑层                             │
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐      │
│  │ 文案生成器   │  │ 标题优化器   │  │ 标签推荐器   │      │
│  │ Generator   │  │ Optimizer   │  │ Tagger      │      │
│  └─────────────┘  └─────────────┘  └─────────────┘      │
└────────────────────┬────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────┐
│                    平台适配层                             │
│  ┌──────────┐ ┌──────────┐ ┌──────────┐ ┌──────────┐   │
│  │ 小红书    │ │ 公众号    │ │ 微博      │ │ 抖音      │   │
│  │ Adapter  │ │ Adapter  │ │ Adapter  │ │ Adapter  │   │
│  └──────────┘ └──────────┘ └──────────┘ └──────────┘   │
└────────────────────┬────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────┐
│                    Prompt 工程层                          │
│  - 平台特定模板                                          │
│  - Few-shot 示例库                                       │
│  - 动态变量注入                                          │
└────────────────────┬────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────┐
│                    AI 模型层                              │
│  - Dashscope API 调用                                    │
│  - 请求重试/降级                                         │
│  - Token 优化                                            │
└─────────────────────────────────────────────────────────┘
```

### 模块划分

```
skills/自媒体文案生成器/
├── index.js              # Skill 入口，命令注册
├── commands/
│   ├── generate.js       # 文案生成命令
│   ├── optimize.js       # 标题优化命令
│   └── tags.js           # 标签推荐命令
├── generators/
│   ├── base.js           # 基础生成器抽象类
│   ├── xiaohongshu.js    # 小红书生成器
│   ├── wechat.js         # 公众号生成器
│   ├── weibo.js          # 微博生成器
│   └── douyin.js         # 抖音生成器
├── optimizers/
│   └── title.js          # 标题优化器
├── taggers/
│   └── index.js          # 标签推荐器
├── adapters/
│   └── platform.js       # 平台格式适配器
├── prompts/
│   ├── xiaohongshu.txt   # 小红书 prompt 模板
│   ├── wechat.txt        # 公众号 prompt 模板
│   ├── weibo.txt         # 微博 prompt 模板
│   ├── douyin.txt        # 抖音 prompt 模板
│   └── title.txt         # 标题优化 prompt
├── examples/
│   └── few-shot.json     # Few-shot 示例库
├── config/
│   └── platforms.yaml    # 平台配置（字数限制、标签规则等）
├── utils/
│   ├── validator.js      # 参数验证
│   ├── formatter.js      # 输出格式化
│   └── logger.js         # 日志工具
└── README.md             # 使用文档
```

---

## 🔌 API 设计

### Skill 命令接口

#### 1. 文案生成
```bash
# 基础用法
/copywriter generate --platform xiaohongshu --topic "夏季防晒霜"

# 完整参数
/copywriter generate \
  --platform xiaohongshu \
  --topic "夏季防晒霜推荐" \
  --desc "SPF50+ 清爽不油腻" \
  --audience "18-30 岁女性" \
  --length medium \
  --tone "亲切真实" \
  --keywords "防晒，油皮，夏季"
```

#### 2. 标题优化
```bash
/copywriter optimize \
  --platform wechat \
  --title "防晒霜怎么选" \
  --direction "click_rate"
```

#### 3. 标签推荐
```bash
/copywriter tags \
  --platform douyin \
  --content "夏季防晒必备..." \
  --goal "exposure"
```

### 程序化 API

```javascript
// 在 Skill 内部调用
const copywriter = require('./generators');

// 生成文案
const result = await copywriter.generate({
  platform: 'xiaohongshu',
  topic: '夏季防晒霜',
  desc: 'SPF50+ 清爽',
  length: 'medium',
  tone: '亲切'
});

// 优化标题
const titles = await copywriter.optimizeTitle({
  platform: 'wechat',
  original: '防晒霜怎么选',
  direction: 'click_rate'
});

// 获取标签
const tags = await copywriter.recommendTags({
  platform: 'douyin',
  content: '...',
  goal: 'exposure'
});
```

### 返回数据结构

```javascript
// 文案生成响应
{
  "success": true,
  "data": {
    "platform": "xiaohongshu",
    "title": "油皮亲妈！这防晒真的不闷痘😭",
    "content": "正文内容...",
    "tags": ["#防晒", "#油皮", "#夏季护肤"],
    "imageSuggestions": ["产品质地特写", "使用前对比"],
    "metadata": {
      "model": "qwen3.5-plus",
      "tokens": 450,
      "generatedAt": "2026-03-02T10:00:00Z"
    }
  }
}

// 标题优化响应
{
  "success": true,
  "data": {
    "original": "防晒霜怎么选",
    "optimized": [
      {
        "title": "3 分钟学会！油皮防晒霜挑选指南",
        "score": 92,
        "reason": "数字吸引 + 痛点明确 + 时效性"
      },
      {
        "title": "防晒霜选不对，越涂越油？",
        "score": 88,
        "reason": "疑问句引发好奇 + 痛点共鸣"
      }
    ]
  }
}

// 标签推荐响应
{
  "success": true,
  "data": {
    "hot": ["#防晒", "#护肤"],
    "precise": ["#油皮防晒", "#清爽不油腻"],
    "longtail": ["#学生党防晒推荐", "#夏季控油防晒"],
    "trending": ["#夏日护肤大作战"]
  }
}
```

---

## 🔐 安全与限流

### API 限流
- 单用户：60 次/分钟
- 单 IP：100 次/分钟
- 批量生成：10 次/小时

### 内容安全
- 敏感词过滤（接入阿里云内容安全）
- 违禁品检测
- 虚假宣传识别

### 数据隐私
- 用户输入不存储（可选开启历史记录）
- Token 加密传输
- 日志脱敏处理

---

## 📊 性能优化

### Prompt 优化
- 模板预编译
- 动态变量缓存
- Few-shot 示例精选（控制 token 数量）

### 请求优化
- 批量请求合并
- 响应流式输出
- 失败重试指数退避

### 缓存策略
```javascript
// 相同参数结果缓存 24 小时
const cacheKey = hash(`${platform}:${topic}:${tone}`);
const cached = await redis.get(cacheKey);
if (cached) return JSON.parse(cached);
```

---

## 🧪 测试策略

| 测试类型 | 工具 | 覆盖率目标 |
|----------|------|------------|
| 单元测试 | Jest | 80%+ |
| 集成测试 | Supertest | 核心流程 100% |
| E2E 测试 | Playwright | 关键用户路径 |
| Prompt 测试 | 人工评审 + A/B 测试 | - |

---

## 📦 部署配置

### 环境变量
```bash
DASHSCOPE_API_KEY=sk-xxx
COPYWRITER_CACHE_TTL=86400
COPYWRITER_RATE_LIMIT=60
COPYWRITER_LOG_LEVEL=info
```

### 配置文件 (config/platforms.yaml)
```yaml
xiaohongshu:
  maxTitleLength: 20
  maxContentLength: 1000
  maxTags: 10
  emojiStyle: abundant
  tone: authentic

wechat:
  maxTitleLength: 64
  maxContentLength: 20000
  maxTags: 0
  emojiStyle: minimal
  tone: professional

weibo:
  maxTitleLength: 0
  maxContentLength: 140
  maxTags: 5
  emojiStyle: moderate
  tone: engaging

douyin:
  maxTitleLength: 30
  maxContentLength: 500
  maxTags: 10
  emojiStyle: moderate
  tone: energetic
```

---

## 🔄 扩展性设计

### 新增平台
1. 在 `config/platforms.yaml` 添加配置
2. 在 `prompts/` 创建新模板
3. 在 `generators/` 实现适配器
4. 在 `index.js` 注册命令

### 自定义模板
```javascript
// 用户可注册自定义模板
copywriter.registerTemplate('my-style', {
  prompt: '...',
  formatter: (output) => '...'
});
```

### 插件系统
```javascript
// 钩子函数
copywriter.hooks.onGenerate.before = async (input) => {...};
copywriter.hooks.onGenerate.after = async (output) => {...};
```
