#!/usr/bin/env python3
"""
Snapmaker Notification Monitor
Watches printer status and triggers notifications on events
"""

import argparse
import json
import time
import sys
import os
from pathlib import Path


def _find_workspace_root() -> Path:
    # Prefer CWD if it looks like a workspace
    cwd = Path.cwd()
    if (cwd / "skills").is_dir():
        return cwd

    d = Path(__file__).resolve().parent
    for _ in range(10):
        if (d / "skills").is_dir():
            return d
        if d == d.parent:
            break
        d = d.parent
    return Path.cwd()


def _default_log_dir() -> Path:
    """Default log dir (portable): <workspace>/snapmaker/logs.

    Falls back to ~/.openclaw/snapmaker/logs if no workspace root is found.
    """
    ws = _find_workspace_root()
    candidate = ws / "snapmaker" / "logs"
    if candidate.parent.exists():
        return candidate
    return Path.home() / ".openclaw" / "snapmaker" / "logs"


def _open_safe_log_file(value: str):
    """Open a log file safely.

    To avoid arbitrary file writes, --log only accepts a *filename* (no paths).
    The file is created under the default log dir.
    """
    if not value:
        raise ValueError("--log requires a filename")

    name = Path(value)
    if name.is_absolute() or len(name.parts) != 1 or "/" in value or "\\" in value:
        raise ValueError("--log must be a filename only (no paths).")

    log_dir = _default_log_dir()
    log_dir.mkdir(parents=True, exist_ok=True)
    return open(log_dir / name.name, "a", encoding="utf-8")

# Import the SnapmakerAPI from snapmaker.py
sys.path.insert(0, str(Path(__file__).parent))
from snapmaker import SnapmakerAPI, format_time


class SnapmakerMonitor:
    def __init__(self, api: SnapmakerAPI, check_interval: int = 30):
        self.api = api
        self.check_interval = check_interval
        self.last_status = None
        self.was_printing = False
        self.last_progress = 0
        
    def check_events(self) -> list:
        """Check for notable events and return list of event descriptions"""
        try:
            status = self.api.get_status()
        except Exception as e:
            return [f"error:connection|Failed to connect to printer: {e}"]
        
        events = []
        
        # Check for print completion
        is_printing = status.get('status') in ['RUNNING', 'PAUSED']
        progress = status.get('progress', 0)
        
        if self.was_printing and not is_printing and progress >= 0.99:
            file_name = status.get('fileName', 'Unknown')
            elapsed = format_time(status.get('elapsedTime', 0))
            events.append(f"success:complete|Print completed: {file_name} (took {elapsed})")
        
        # Check for new print starting
        if not self.was_printing and is_printing:
            file_name = status.get('fileName', 'Unknown')
            est_time = format_time(status.get('estimatedTime', 0))
            events.append(f"info:started|Print started: {file_name} (est. {est_time})")
        
        # Check for filament out
        if status.get('isFilamentOut'):
            events.append(f"warning:filament|Filament out detected!")
        
        # Check for enclosure door opened during print
        if is_printing and status.get('isEnclosureDoorOpen'):
            events.append(f"warning:door|Enclosure door opened during print")
        
        # Check for pause
        if status.get('status') == 'PAUSED' and self.last_status and self.last_status.get('status') == 'RUNNING':
            events.append(f"warning:paused|Print paused")
        
        # Check for resume
        if status.get('status') == 'RUNNING' and self.last_status and self.last_status.get('status') == 'PAUSED':
            events.append(f"info:resumed|Print resumed")
        
        # Check for significant progress milestones (25%, 50%, 75%)
        if is_printing:
            for milestone in [0.25, 0.50, 0.75]:
                if self.last_progress < milestone <= progress:
                    percent = int(milestone * 100)
                    remaining = format_time(status.get('remainingTime', 0))
                    events.append(f"info:progress|Print {percent}% complete (est. {remaining} remaining)")
        
        # Update state
        self.last_status = status
        self.was_printing = is_printing
        self.last_progress = progress
        
        return events
    
    def run(self, callback=None):
        """Run monitoring loop"""
        print(f"Monitoring printer at {self.api.ip}...")
        print(f"Check interval: {self.check_interval} seconds")
        print("Press Ctrl+C to stop\n")
        
        try:
            while True:
                events = self.check_events()
                
                for event in events:
                    timestamp = time.strftime('%Y-%m-%d %H:%M:%S')
                    print(f"[{timestamp}] {event}")
                    
                    if callback:
                        callback(event)
                
                time.sleep(self.check_interval)
        
        except KeyboardInterrupt:
            print("\nMonitoring stopped")


def main():
    parser = argparse.ArgumentParser(
        description='Monitor Snapmaker for events',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Event Format:
  level:type|message
  
  Levels: success, info, warning, error
  Types: complete, started, filament, door, paused, resumed, progress, connection
  
Examples:
  %(prog)s                    # Monitor with default settings
  %(prog)s --interval 10      # Check every 10 seconds
  %(prog)s --log events.log   # Write events to <workspace>/snapmaker/logs/events.log
        """
    )
    
    parser.add_argument('--config', help='Path to config file')
    parser.add_argument('--interval', type=int, default=30,
                       help='Check interval in seconds (default: 30)')
    parser.add_argument('--log', help='Log events to file')
    
    args = parser.parse_args()
    
    # Initialize API
    try:
        api = SnapmakerAPI(args.config)
    except Exception as e:
        print(f"Error loading config: {e}", file=sys.stderr)
        sys.exit(1)
    
    # Setup logging
    log_file = None
    if args.log:
        try:
            log_file = _open_safe_log_file(args.log)
        except Exception as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(2)
        
        def log_callback(event):
            timestamp = time.strftime('%Y-%m-%d %H:%M:%S')
            log_file.write(f"[{timestamp}] {event}\n")
            log_file.flush()
    else:
        log_callback = None
    
    # Run monitor
    monitor = SnapmakerMonitor(api, args.interval)
    try:
        monitor.run(callback=log_callback)
    finally:
        if log_file:
            log_file.close()


if __name__ == '__main__':
    main()
