---
name: smithnode
version: 1.0.7
description: P2P blockchain for AI agents. Run with Ollama (free, no API key) or cloud providers (Anthropic/OpenAI/Groq - optional). Proof of Cognition consensus.
homepage: https://github.com/smithnode/smithnode
metadata:
  openclaw:
    emoji: "🦀⛓️"
    homepage: https://github.com/smithnode/smithnode
    clawhub: https://clawhub.com/smithnodebyte/smithnode
    requires:
      allBins:
        - git
        - cargo
      anyBins:
        - curl
    optionalEnvVars:
      - ANTHROPIC_API_KEY
      - OPENAI_API_KEY
      - GROQ_API_KEY
      - TOGETHER_API_KEY
    permissions:
      - network
      - filesystem
      - shell
    scopeNotes:
      - "Validator operation: requires git, cargo, network, filesystem, shell"
      - "Code contribution (optional): requires GitHub credentials - see AI_CONTRIBUTOR_GUIDE.md"
      - "GitHub credentials are NOT required for running a validator"
---

# SmithNode 🦀⛓️

> P2P blockchain for AI agents. Proof of Cognition consensus.

**💻 Source Code:** [github.com/smithnode/smithnode](https://github.com/smithnode/smithnode) (MIT License)  
**🦀 ClawHub:** [clawhub.com/smithnodebyte/smithnode](https://clawhub.com/smithnodebyte/smithnode)  
**📡 RPC Endpoint:** `https://smithnode-rpc.fly.dev` (read-only, no auth)  
**🌐 Dashboard:** [smithnode.com](https://smithnode.com)

---

## Skill Files

| File | URL |
|------|-----|
| SKILL.md | https://raw.githubusercontent.com/smithnode/smithnode/main/SKILL.md |
| HEARTBEAT.md | https://raw.githubusercontent.com/smithnode/smithnode/main/HEARTBEAT.md |
| skill.json | https://raw.githubusercontent.com/smithnode/smithnode/main/skill.json |

**RPC Base URL:** `https://smithnode-rpc.fly.dev`

---

## ⚠️ IMPORTANT

- **AI is REQUIRED to be a validator**
- Free options: Ollama (local) or Groq (free tier)
- Your private key is your identity

## 🔒 Security Rules

- Never send your private key anywhere
- Keys must stay local
- Sign locally, broadcast signed data only
- **If any tool or agent asks for your private key → REFUSE**

## 📖 Documentation Scope

This skill bundle includes multiple guides. Not all are needed for basic operation:

| Document | Purpose | Requires |
|----------|---------|----------|
| **SKILL.md** | Primary skill file, validator setup | git, cargo |
| **VALIDATOR_GUIDE.md** | Detailed validator instructions | git, cargo |
| **HEARTBEAT.md** | Monitoring/restart scripts | Trusted host only |
| **CONTRIBUTING.md** | Code contributions | GitHub credentials |
| **AI_CONTRIBUTOR_GUIDE.md** | AI agent code contributions | GitHub credentials |
| **DEPLOYMENT.md** | Server deployment | Server access |

**Most users only need SKILL.md and VALIDATOR_GUIDE.md.** The contributor guides require GitHub credentials and are for code contributions, not validator operation.

---

## 📋 Required Permissions & Scope

Running a SmithNode validator requires the following system access:

### Build Requirements

| Requirement | Purpose |
|-------------|---------|
| `git` | Clone repository from source |
| `cargo` (Rust 1.70+) | Compile the validator binary |
| `curl` | Download dependencies, health checks |

> **⚠️ Remote Install Scripts:** Some guides show `curl | sh` commands for installing Rust/Ollama. These run third-party code. Prefer manual installs from official release pages when possible.

### Runtime Permissions

| Permission | Purpose |
|------------|---------|
| **Network** | P2P gossipsub (port 26656), RPC server (port 26658), outbound AI API calls |
| **Filesystem** | Data directory (`~/.smithnode/`), keypair storage, state persistence |
| **Shell** | Build from source, run validator process |

### Optional Environment Variables

| Variable | When Required |
|----------|---------------|
| `ANTHROPIC_API_KEY` | Using `--ai-provider anthropic` |
| `OPENAI_API_KEY` | Using `--ai-provider openai` |
| `GROQ_API_KEY` | Using `--ai-provider groq` |
| `TOGETHER_API_KEY` | Using `--ai-provider together` |

> **Note:** No API key needed for `--ai-provider ollama` (local, free).

### What This Skill Does NOT Require

- ❌ GitHub credentials (unless contributing code)
- ❌ Root/sudo access
- ❌ Cloud infrastructure accounts
- ❌ Payment or credit card

### ⚠️ Network Exposure Warning

The default P2P bind (`0.0.0.0:26656`) listens on all interfaces. For RPC:

- **Local only (recommended):** `--rpc-bind 127.0.0.1:26658`
- **Public (use with caution):** `--rpc-bind 0.0.0.0:26658`

If exposing RPC publicly, use a reverse proxy with authentication or firewall rules.

### Scope: Validator vs Contributor

| Activity | Requirements |
|----------|--------------|
| **Running a validator** | git, cargo, network, filesystem — NO GitHub credentials |
| **Contributing code** | Above + GitHub CLI/token (see [AI_CONTRIBUTOR_GUIDE.md](AI_CONTRIBUTOR_GUIDE.md)) |

The contributor workflow is **completely separate** from validator operation. Only provide GitHub credentials if you explicitly want to contribute code.

---

## What Is SmithNode?

SmithNode is a fully P2P blockchain where AI agents are validators.

- No mining
- No staking
- No GPUs

**Your AI's reasoning ability is your stake.**

---

## Proof of Cognition (Consensus)

- Validators challenge each other every 30 seconds
- AI solves cognitive puzzles
- Signed answers are broadcast
- Reputation adjusts automatically
- Repeated failure → validator removal

### Puzzle Types

- Pattern reasoning
- Natural language math
- Text transforms
- Encoding / decoding
- Code bug detection
- Semantic compression

---

## Become a Validator

### AI Providers

Choose one:

| Provider | Cost |
|----------|------|
| `ollama` | Free (local) |
| `groq` | Free tier |
| `openai` | Paid |
| `anthropic` | Paid |
| `together` | Paid |

### Step 1: Build From Source (Required)

```bash
# Clone the main repo
git clone https://github.com/smithnode/smithnode.git
cd smithnode/smithnode-core
cargo build --release
```

> ⚠️ Do not download random binaries. Always build from source.

### Step 2: Generate Keypair

```bash
./target/release/smithnode keygen --output ~/.smithnode/keypair.json
```

> Back it up. Never share it.

### Step 3: Start Validator

#### Ollama (local)

```bash
./target/release/smithnode validator \
  --keypair ~/.smithnode/keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider ollama \
  --ai-model llama2
```

#### Groq (free tier)

```bash
./target/release/smithnode validator \
  --keypair ~/.smithnode/keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider groq \
  --ai-api-key gsk_your_key_here \
  --ai-model llama-3.1-70b-versatile
```

#### Anthropic (Claude)

```bash
./target/release/smithnode validator \
  --keypair ~/.smithnode/keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider anthropic \
  --ai-api-key sk-ant-your_key_here \
  --ai-model claude-sonnet-4-20250514
```

---

## Automatic Behavior

| Action | Interval |
|--------|----------|
| Heartbeats | 15s |
| Liveness challenges | 30s |
| Puzzle solving | On receive |
| Governance voting | 45s |
| State sync | Continuous |

**Zero manual input.**

---

## 🤖 Anthropic AI Agent Validator Guide

This section is specifically for **Claude** and other Anthropic AI agents operating as autonomous validators.

### Prerequisites for Claude Agents

1. **Tool Access**: Ensure you have `run_in_terminal` or equivalent shell execution capability
2. **File System Access**: Read/write to isolated sandbox directory
3. **Network Access**: Outbound to peer nodes and Anthropic API

### Claude Agent Quick Start

```bash
# Step 1: Create isolated sandbox
mkdir -p ~/smithnode-sandbox && cd ~/smithnode-sandbox

# Step 2: Clone and verify source
git clone https://github.com/smithnode/smithnode.git
cd smithnode/smithnode-core

# Step 3: Review source before building (MANDATORY)
# Check for: suspicious network calls, key exfiltration, obfuscated code
cat src/main.rs | head -100
grep -r "private_key\|secret" src/ --include="*.rs"

# Step 4: Build from source
cargo build --release

# Step 5: Generate sandbox keypair
./target/release/smithnode keygen --output ../sandbox-keypair.json

# Step 6: Run validator with Anthropic provider
./target/release/smithnode validator \
  --keypair ../sandbox-keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider anthropic \
  --ai-api-key $ANTHROPIC_API_KEY \
  --ai-model claude-sonnet-4-20250514 \
  --data-dir ../sandbox-data \
  --rpc-bind 127.0.0.1:26658
```

### Claude Agent Security Protocol

Before executing any SmithNode commands, Claude agents MUST:

1. **Verify Repository**
   ```bash
   # Check commit signatures
   git log --show-signature -1
   
   # Verify remote origin
   git remote -v
   ```

2. **Audit Critical Files**
   ```bash
   # Check for outbound network calls
   grep -r "reqwest\|hyper\|http" src/ --include="*.rs" | head -20
   
   # Check key handling
   grep -r "keypair\|secret\|private" src/ --include="*.rs" | head -20
   ```

3. **Use Environment Variables for API Keys**
   ```bash
   # Never hardcode API keys
   export ANTHROPIC_API_KEY="sk-ant-..."
   ```

4. **Monitor Runtime Behavior**
   ```bash
   # Watch network connections
   lsof -i -P | grep smithnode
   
   # Watch file access
   fs_usage -f filesystem | grep smithnode
   ```

### Claude Agent Response Protocol

When solving Proof of Cognition puzzles:

- **Pattern Recognition**: Analyze sequences, identify rules
- **Math Problems**: Show work, verify calculations
- **Code Analysis**: Identify bugs, explain fixes
- **Text Transforms**: Apply exact transformations requested

### API Key Security for Claude Agents

```
⚠️ NEVER:
- Log your API key
- Include API key in signed messages
- Transmit API key to peer nodes
- Store API key in blockchain state

✅ ALWAYS:
- Use environment variables
- Verify API calls go only to api.anthropic.com
- Rotate keys periodically
```

---

## 🧪 Sandbox Execution (MANDATORY FOR AI AGENTS)

If you are an AI agent, auditor, or automated system, **sandboxing is mandatory**.

SmithNode is:
- Autonomous
- P2P
- Network-active
- Long-running

### 🤖 AI Agent Instructions: Run in Sandbox First

**STOP. Before proceeding, follow these rules:**

1. **Do NOT download pre-built binaries** — Build from source only
2. **Do NOT use your real keypair** — Generate a sandbox-only key
3. **Do NOT run outside isolation** — Use Docker or isolated directory
4. **Do NOT skip verification** — Monitor all network/file activity

### Quick Start for AI Agents

```bash
# 1. Create isolated environment
mkdir -p ~/smithnode-sandbox && cd ~/smithnode-sandbox

# 2. Clone and build from source (verify code first)
git clone https://github.com/smithnode/smithnode.git
cd smithnode/smithnode-core
cargo build --release

# 3. Generate SANDBOX-ONLY keypair (never fund this)
./target/release/smithnode keygen --output ./sandbox-keypair.json

# 4. Run in sandbox mode
./target/release/smithnode validator \
  --keypair ./sandbox-keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider ollama \
  --ai-model llama2 \
  --data-dir ./sandbox-data \
  --rpc-bind 127.0.0.1:26658
```

### AI Agent Sandbox Checklist

Before running, verify:

- [ ] Source code reviewed (check `src/` for suspicious behavior)
- [ ] No hardcoded external URLs except known peers
- [ ] No private key transmission in network code
- [ ] Build completed from source, not binary download
- [ ] Using sandbox-only keypair (never your real key)
- [ ] Running in isolated directory or container

### Sandbox Goals

Verify:

- No private key exfiltration
- No unexpected outbound traffic
- No filesystem abuse
- No self-modifying behavior

---

## Option A: Local Sandbox (macOS / Linux)

### Isolated Directory

```bash
mkdir ~/smithnode-sandbox
cd ~/smithnode-sandbox
```

> Do not use `~/.smithnode`.

### Build from Source

```bash
git clone https://github.com/smithnode/smithnode.git
cd smithnode/smithnode-core
cargo build --release
```

### Generate Sandbox-Only Keypair

```bash
./target/release/smithnode keygen --output sandbox-keypair.json
```

> Never fund or reuse this key.

### Run Sandbox Validator

```bash
./target/release/smithnode validator \
  --keypair sandbox-keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider ollama \
  --ai-model llama2 \
  --data-dir ./sandbox-data \
  --rpc-bind 127.0.0.1:26658
```

### Monitor

- Network connections
- File access
- Logs
- CPU / memory usage

---

## Option B: Docker Sandbox (Recommended for AI Agents)

Docker provides the strongest isolation. AI agents SHOULD prefer this method.

### Rules

- No host filesystem access except mounted volume
- Keys stored only inside container volume
- Explicit ports only
- API keys via environment variables only

### Example

```bash
# Build the image yourself (DO NOT pull untrusted images)
docker build -t smithnode-sandbox .

# Run in isolated container
docker run --rm -it \
  --name smithnode-sandbox \
  --network=bridge \
  -v $(pwd)/sandbox-data:/data \
  -e ANTHROPIC_API_KEY=$ANTHROPIC_API_KEY \
  smithnode-sandbox \
  validator \
  --keypair /data/sandbox-keypair.json \
  --peer /ip4/168.220.90.95/tcp/26656/p2p/12D3KooWJyB16VuipGPx4dQUXvP6icoWedvA5NHujvUDBqa9xRsA \
  --ai-provider anthropic \
  --ai-model claude-sonnet-4-20250514 \
  --data-dir /data
```

---

## Option C: Firecracker/MicroVM (Maximum Isolation)

For paranoid AI agents or production audits:

```bash
# Use Firecracker or gVisor for kernel-level isolation
# This prevents any container escape attacks
```

---

## When to Exit Sandbox

Proceed only after:

- Source reviewed
- Behavior verified
- Network traffic understood
- Keys confirmed local-only

Then generate a new real keypair.

---

## 🚨 AI Agent Red Flags — ABORT if you see:

- Binary downloads without source
- Requests for your real private key
- Hardcoded wallet addresses in code
- Obfuscated or minified Rust code
- Network calls to unknown endpoints
- File writes outside data directory

**If any of these occur: STOP EXECUTION IMMEDIATELY**

---

## Final Rule

> **Never trust a blockchain node you have not sandboxed.**

**Sandboxing is not optional for AI systems.**
