#!/usr/bin/env node

/**
 * SlideSpeak API Helper Script
 *
 * A Node.js script for interacting with the SlideSpeak API.
 * Uses native fetch (Node 18+) - no dependencies required.
 * Handles async task polling automatically with configurable timeout.
 */

const API_BASE = 'https://api.slidespeak.co/api/v1';
const API_KEY = process.env.SLIDESPEAK_API_KEY;
const DEFAULT_TIMEOUT = 120000; // 120 seconds
const POLL_INTERVAL = 2000; // 2 seconds

// Helper to output JSON response
function output(success, data, error = null) {
  console.log(JSON.stringify(success ? { success: true, data } : { success: false, error }, null, 2));
  process.exit(success ? 0 : 1);
}

// Check API key
function checkApiKey() {
  if (!API_KEY) {
    output(false, null, 'SLIDESPEAK_API_KEY environment variable is not set');
  }
}

// Make API request
async function apiRequest(endpoint, options = {}) {
  const url = endpoint.startsWith('http') ? endpoint : `${API_BASE}${endpoint}`;
  const headers = {
    'X-API-Key': API_KEY,
    ...options.headers,
  };

  if (options.json) {
    headers['Content-Type'] = 'application/json';
    options.body = JSON.stringify(options.json);
    delete options.json;
  }

  const response = await fetch(url, { ...options, headers });

  if (!response.ok) {
    const text = await response.text();
    throw new Error(`API request failed (${response.status}): ${text}`);
  }

  return response.json();
}

// Poll task until completion
// Returns { complete: true, ...result } on success
// Returns { complete: false, task_id, message } on timeout (so caller can retry later)
async function pollTask(taskId, timeout = DEFAULT_TIMEOUT) {
  const startTime = Date.now();
  let pollCount = 0;
  let lastStatus = null;

  while (Date.now() - startTime < timeout) {
    pollCount++;
    const elapsed = Math.round((Date.now() - startTime) / 1000);
    process.stderr.write(`\rPolling task ${taskId.slice(0, 8)}... (${elapsed}s, attempt ${pollCount})`);

    const result = await apiRequest(`/task_status/${taskId}`);
    lastStatus = result.task_status;

    if (result.task_status === 'SUCCESS') {
      process.stderr.write('\n');
      return { complete: true, ...result };
    }

    if (result.task_status === 'FAILURE' || result.task_status === 'ERROR') {
      process.stderr.write('\n');
      throw new Error(`Task failed: ${result.task_result || result.task_info || 'Unknown error'}`);
    }

    // Still processing, wait before next poll
    await new Promise(resolve => setTimeout(resolve, POLL_INTERVAL));
  }

  process.stderr.write('\n');
  // Return task info instead of throwing, so the agent can poll later
  return {
    complete: false,
    task_id: taskId,
    task_status: lastStatus || 'STARTED',
    message: `Task still processing after ${timeout / 1000}s. Check status with: node scripts/slidespeak.mjs status ${taskId}`,
  };
}

// Parse command line arguments
function parseArgs(args) {
  const result = { _: [] };
  let i = 0;

  while (i < args.length) {
    const arg = args[i];

    if (arg.startsWith('--')) {
      const key = arg.slice(2);

      if (key.startsWith('no-')) {
        result[key.slice(3)] = false;
        i++;
      } else if (i + 1 < args.length && !args[i + 1].startsWith('--')) {
        result[key] = args[i + 1];
        i += 2;
      } else {
        result[key] = true;
        i++;
      }
    } else {
      result._.push(arg);
      i++;
    }
  }

  return result;
}

// Commands
const commands = {
  async me() {
    checkApiKey();
    const result = await apiRequest('/me');
    output(true, result);
  },

  async generate(args) {
    checkApiKey();

    const body = {};

    if (args.text) {
      body.plain_text = args.text;
    } else if (args.document) {
      body.document_uuids = [args.document];
    } else {
      output(false, null, 'Either --text or --document is required');
    }

    if (args.length) body.length = parseInt(args.length, 10);
    if (args.template) body.template = args.template;
    if (args.language) body.language = args.language;
    if (args.tone) body.tone = args.tone;
    if (args.verbosity) body.verbosity = args.verbosity;
    if (args.images === false) body.fetch_images = false;
    if (args.cover === false) body.include_cover = false;
    if (args.toc === false) body.include_table_of_contents = false;

    const result = await apiRequest('/presentation/generate', {
      method: 'POST',
      json: body,
    });

    // If --no-wait, return task_id immediately
    if (args.wait === false) {
      output(true, {
        task_id: result.task_id,
        message: `Task started. Check status with: node scripts/slidespeak.mjs status ${result.task_id}`,
      });
      return;
    }

    // Poll for completion
    const completed = await pollTask(result.task_id, parseInt(args.timeout, 10) || DEFAULT_TIMEOUT);
    output(true, completed);
  },

  async 'generate-slides'(args) {
    checkApiKey();

    if (!args.config) {
      output(false, null, '--config is required (path to JSON file)');
    }

    const fs = await import('fs');
    const configContent = fs.readFileSync(args.config, 'utf-8');
    const config = JSON.parse(configContent);

    const result = await apiRequest('/presentation/generate/slide-by-slide', {
      method: 'POST',
      json: config,
    });

    if (args.wait === false) {
      output(true, {
        task_id: result.task_id,
        message: `Task started. Check status with: node scripts/slidespeak.mjs status ${result.task_id}`,
      });
      return;
    }

    const completed = await pollTask(result.task_id, parseInt(args.timeout, 10) || DEFAULT_TIMEOUT);
    output(true, completed);
  },

  async upload(args) {
    checkApiKey();

    const filePath = args._[0];
    if (!filePath) {
      output(false, null, 'File path is required');
    }

    const fs = await import('fs');
    const path = await import('path');

    if (!fs.existsSync(filePath)) {
      output(false, null, `File not found: ${filePath}`);
    }

    const fileBuffer = fs.readFileSync(filePath);
    const fileName = path.basename(filePath);

    // Create form data manually for native fetch
    const boundary = '----FormBoundary' + Math.random().toString(36).slice(2);
    const bodyParts = [
      `--${boundary}`,
      `Content-Disposition: form-data; name="file"; filename="${fileName}"`,
      'Content-Type: application/octet-stream',
      '',
      fileBuffer.toString('binary'),
      `--${boundary}--`,
    ];

    const body = bodyParts.join('\r\n');

    const response = await fetch(`${API_BASE}/document/upload`, {
      method: 'POST',
      headers: {
        'X-API-Key': API_KEY,
        'Content-Type': `multipart/form-data; boundary=${boundary}`,
      },
      body: Buffer.from(body, 'binary'),
    });

    if (!response.ok) {
      const text = await response.text();
      throw new Error(`Upload failed (${response.status}): ${text}`);
    }

    const result = await response.json();

    if (args.wait === false) {
      output(true, {
        task_id: result.task_id,
        message: `Upload started. Check status with: node scripts/slidespeak.mjs status ${result.task_id}`,
      });
      return;
    }

    // Poll for completion
    const completed = await pollTask(result.task_id, parseInt(args.timeout, 10) || DEFAULT_TIMEOUT);
    output(true, completed);
  },

  async status(args) {
    checkApiKey();

    const taskId = args._[0];
    if (!taskId) {
      output(false, null, 'Task ID is required');
    }

    const result = await apiRequest(`/task_status/${taskId}`);
    output(true, result);
  },

  async download(args) {
    checkApiKey();

    const requestId = args._[0];
    if (!requestId) {
      output(false, null, 'Request ID is required');
    }

    const result = await apiRequest(`/presentation/download/${requestId}`);
    output(true, result);
  },

  async templates(args) {
    checkApiKey();

    const endpoint = args.branded ? '/presentation/templates/branded' : '/presentation/templates';
    const result = await apiRequest(endpoint);
    output(true, result);
  },

  async 'edit-slide'(args) {
    checkApiKey();

    if (!args['presentation-id']) {
      output(false, null, '--presentation-id is required');
    }
    if (!args.type) {
      output(false, null, '--type is required (INSERT, REGENERATE, or REMOVE)');
    }
    if (args.position === undefined) {
      output(false, null, '--position is required');
    }

    const editType = args.type.toUpperCase();
    if (!['INSERT', 'REGENERATE', 'REMOVE'].includes(editType)) {
      output(false, null, '--type must be INSERT, REGENERATE, or REMOVE');
    }

    if (editType !== 'REMOVE' && !args.prompt) {
      output(false, null, '--prompt is required for INSERT and REGENERATE');
    }

    const body = {
      presentation_id: args['presentation-id'],
      edit_type: editType,
      position: parseInt(args.position, 10),
    };

    if (args.prompt) body.prompt = args.prompt;
    if (args.verbosity) body.verbosity = args.verbosity;
    if (args.tone) body.tone = args.tone;
    if (args['speaker-notes']) body.add_speaker_notes = true;

    const result = await apiRequest('/presentation/edit/slide', {
      method: 'POST',
      json: body,
    });

    if (args.wait === false) {
      output(true, {
        task_id: result.task_id,
        message: `Edit started. Check status with: node scripts/slidespeak.mjs status ${result.task_id}`,
      });
      return;
    }

    const completed = await pollTask(result.task_id, parseInt(args.timeout, 10) || DEFAULT_TIMEOUT);
    output(true, completed);
  },

  async 'webhook-subscribe'(args) {
    checkApiKey();

    if (!args.url) {
      output(false, null, '--url is required');
    }

    const result = await apiRequest('/webhook/subscribe', {
      method: 'POST',
      json: { url: args.url },
    });
    output(true, result);
  },

  async 'webhook-unsubscribe'(args) {
    checkApiKey();

    if (!args.url) {
      output(false, null, '--url is required');
    }

    const result = await apiRequest('/webhook/unsubscribe', {
      method: 'DELETE',
      json: { url: args.url },
    });
    output(true, result);
  },

  help() {
    console.log(`
SlideSpeak API Helper

Usage: node slidespeak.mjs <command> [options]

Commands:
  me                          Get account info
  generate                    Generate a presentation (takes 30-60s)
    --text <text>             Topic or content (required if no --document)
    --document <uuid>         Document UUID (required if no --text)
    --length <n>              Number of slides (default: 10)
    --template <name>         Template name or ID
    --language <lang>         Output language
    --tone <tone>             casual, professional, funny, educational, sales_pitch
    --verbosity <level>       concise, standard, text-heavy
    --no-images               Disable stock images
    --no-cover                Exclude cover slide
    --no-toc                  Exclude table of contents
    --no-wait                 Return task_id immediately without waiting
    --timeout <ms>            Polling timeout (default: 120000)

  generate-slides             Generate slide-by-slide (takes 30-60s)
    --config <file>           Path to JSON config file
    --no-wait                 Return task_id immediately without waiting

  upload <file>               Upload a document
    --no-wait                 Return task_id immediately without waiting
    --timeout <ms>            Polling timeout

  status <task_id>            Check task status (use to poll incomplete tasks)

  download <request_id>       Get download URL

  templates                   List templates
    --branded                 List branded templates

  edit-slide                  Edit a slide
    --presentation-id <id>    Presentation ID (required)
    --type <type>             INSERT, REGENERATE, or REMOVE (required)
    --position <n>            Slide position (required)
    --prompt <text>           Content prompt (required for INSERT/REGENERATE)
    --verbosity <level>       concise, standard, text-heavy
    --tone <tone>             Tone for content
    --no-wait                 Return task_id immediately without waiting

  webhook-subscribe           Subscribe to webhooks
    --url <url>               Webhook URL (required)

  webhook-unsubscribe         Unsubscribe from webhooks
    --url <url>               Webhook URL (required)

  help                        Show this help

Environment:
  SLIDESPEAK_API_KEY          Your SlideSpeak API key (required)

Notes:
  - Generation takes 30-60 seconds. The script waits by default.
  - Use --no-wait to return immediately with a task_id.
  - Use 'status <task_id>' to check progress of incomplete tasks.
  - If a task times out, the response includes the task_id for later polling.
`);
    process.exit(0);
  },
};

// Main
async function main() {
  const args = parseArgs(process.argv.slice(2));
  const command = args._[0];
  args._ = args._.slice(1);

  if (!command || command === 'help' || args.help) {
    commands.help();
  }

  if (!commands[command]) {
    output(false, null, `Unknown command: ${command}. Run with 'help' for usage.`);
  }

  try {
    await commands[command](args);
  } catch (error) {
    output(false, null, error.message);
  }
}

main();
