const { execSync } = require("node:child_process");

function ensureSession() {
  try {
    execSync('tmux has-session -t claw', { stdio: "ignore" });
  } catch {
    execSync('tmux new -s claw -d');
  }
}

function sendCommand(cmd) {
  const escaped = cmd.replace(/"/g, '\\"');
  execSync(`tmux send-keys -t claw "${escaped}" C-m`);
}

function readOutput() {
  try {
    const buf = execSync('tmux capture-pane -t claw -p -S -200');
    return buf.toString("utf8");
  } catch (e) {
    return `ERROR READING TMUX OUTPUT: ${e.message}`;
  }
}

function isDangerous(cmd) {
  const bad = ["sudo", " rm ", " rm-", "reboot", "shutdown", "mkfs", "dd "];
  const lower = ` ${cmd.toLowerCase()} `;
  return bad.some(k => lower.includes(k));
}

// MAIN ENTRYPOINT
// OpenClaw will call this function when using the skill tool
async function claw_shell_run(input) {
  const { command } = input;
  if (!command || typeof command !== "string") {
    return { error: "command is required" };
  }

  if (isDangerous(command)) {
    return {
      error: "dangerous_command",
      message: `Command looks dangerous. Ask the user for explicit approval before running: ${command}`
    };
  }

  ensureSession();
  sendCommand(command);

  // small delay so command can run
  await new Promise(r => setTimeout(r, 500));

  const output = readOutput();
  return { command, output };
}

module.exports = {
  claw_shell_run,
};