---
name: skill-writer
description: Write high-quality agent skills (SKILL.md files) for ClawdHub/MoltHub. Use when creating a new skill from scratch, structuring skill content, writing effective frontmatter and descriptions, choosing section patterns, or following best practices for agent-consumable technical documentation.
metadata: {"clawdbot":{"emoji":"✍️","requires":{"anyBins":["npx"]},"os":["linux","darwin","win32"]}}
---

# Skill Writer

Write well-structured, effective SKILL.md files for the ClawdHub registry. Covers the skill format specification, frontmatter schema, content patterns, example quality, and common anti-patterns.

## When to Use

- Creating a new skill from scratch
- Structuring technical content as an agent skill
- Writing frontmatter that the registry indexes correctly
- Choosing section organization for different skill types
- Reviewing your own skill before publishing

## The SKILL.md Format

A skill is a single Markdown file with YAML frontmatter. The agent loads it on demand and follows its instructions.

```markdown
---
name: my-skill-slug
description: One-sentence description of when to use this skill.
metadata: {"clawdbot":{"emoji":"🔧","requires":{"anyBins":["tool1","tool2"]},"os":["linux","darwin","win32"]}}
---

# Skill Title

One-paragraph summary of what this skill covers.

## When to Use

- Bullet list of trigger scenarios

## Main Content Sections

### Subsection with examples

Code blocks, commands, patterns...

## Tips

- Practical advice bullets
```

## Frontmatter Schema

### `name` (required)

The skill's slug identifier. Must match what you publish with.

```yaml
name: my-skill
```

Rules:
- Lowercase, hyphenated: `csv-pipeline`, `git-workflows`
- No spaces, no underscores
- Keep it short and descriptive (1-3 words)
- Check for slug collisions before publishing: `npx molthub@latest search "your-slug"`

### `description` (required)

The single most important field. This is what:
1. The registry indexes for semantic search (vector embeddings)
2. The agent reads to decide whether to activate the skill
3. Users see when browsing search results

```yaml
# GOOD: Specific triggers and scope
description: Write Makefiles for any project type. Use when setting up build automation, defining multi-target builds, managing dependencies between tasks, creating project task runners, or using Make for non-C projects (Go, Python, Docker, Node.js). Also covers Just and Task as modern alternatives.

# BAD: Vague, no triggers
description: A skill about Makefiles.

# BAD: Too long (gets truncated in search results)
description: This skill covers everything you need to know about Makefiles including variables, targets, prerequisites, pattern rules, automatic variables, phony targets, conditional logic, multi-directory builds, includes, silent execution, and also covers Just and Task as modern alternatives to Make for projects that use Go, Python, Docker, or Node.js...
```

Pattern for effective descriptions:
```
[What it does]. Use when [trigger 1], [trigger 2], [trigger 3]. Also covers [related topic].
```

### `metadata` (required)

JSON object with the `clawdbot` schema:

```yaml
metadata: {"clawdbot":{"emoji":"🔧","requires":{"anyBins":["make","just"]},"os":["linux","darwin","win32"]}}
```

Fields:
- **`emoji`**: Single emoji displayed in registry listings
- **`requires.anyBins`**: Array of CLI tools the skill needs (at least one must be available)
- **`os`**: Array of supported platforms: `"linux"`, `"darwin"` (macOS), `"win32"` (Windows)

Choose `requires.anyBins` carefully:
```yaml
# Good: lists the actual tools the skill's commands use
"requires": {"anyBins": ["docker", "docker-compose"]}

# Bad: lists generic tools every system has
"requires": {"anyBins": ["bash", "echo"]}

# Good for skills that work via multiple tools
"requires": {"anyBins": ["make", "just", "task"]}
```

## Content Structure

### The "When to Use" Section

Always include this immediately after the title paragraph. It tells the agent (and the user) the specific scenarios where this skill applies.

```markdown
## When to Use

- Automating build, test, lint, deploy commands
- Defining dependencies between tasks (build before test)
- Creating a project-level task runner
- Replacing long CLI commands with short targets
```

Rules:
- 4-8 bullet points
- Each bullet is a concrete scenario, not an abstract concept
- Start with a verb or gerund: "Automating...", "Debugging...", "Converting..."
- Don't repeat the description field verbatim

### Main Content Sections

Organize by task, not by concept. The agent needs to find the right command for a specific situation.

```markdown
## GOOD: Organized by task
## Encode and Decode
### Base64
### URL Encoding
### Hex

## BAD: Organized by abstraction
## Theory of Encoding
## Encoding Types
## Advanced Topics
```

### Code Blocks

Every section should have at least one code block. Skills without code blocks are opinions, not tools.

````markdown
## GOOD: Concrete, runnable example
```bash
# Encode a string to Base64
echo -n "Hello, World!" | base64
# SGVsbG8sIFdvcmxkIQ==
```

## BAD: Abstract description
Base64 encoding converts binary data to ASCII text using a 64-character alphabet...
````

Code block best practices:
- **Always specify the language** (`bash`, `python`, `javascript`, `yaml`, `sql`, etc.)
- **Show the output** in a comment below the command
- **Use realistic values**, not `foo`/`bar` (use `myapp`, `api-server`, real IP formats)
- **Include the most common case first**, then variations
- **Add inline comments** for non-obvious flags or arguments

### Multi-Language Coverage

If a skill applies across languages, use consistent section structure:

```markdown
## Hashing

### Bash
```bash
echo -n "Hello" | sha256sum
```

### JavaScript
```javascript
const crypto = require('crypto');
crypto.createHash('sha256').update('Hello').digest('hex');
```

### Python
```python
import hashlib
hashlib.sha256(b"Hello").hexdigest()
```
```

Order: Bash first (most universal), then by popularity for the topic.

### The "Tips" Section

End every skill with a Tips section. These are the distilled wisdom — the things that save hours of debugging.

```markdown
## Tips

- The number one Makefile bug: using spaces instead of tabs for indentation.
- SHA-256 is the standard for integrity checks. MD5 is fine for dedup but broken for cryptographic use.
- Never schedule critical cron jobs between 1:00-3:00 AM if DST applies.
```

Rules:
- 5-10 bullets
- Each tip is a standalone insight (no dependencies on other tips)
- Prioritize gotchas and non-obvious behavior over basic advice
- No "always use best practices" platitudes

## Skill Types and Templates

### CLI Tool Reference

For skills about a specific tool or command family.

```markdown
---
name: tool-name
description: [What tool does]. Use when [scenario 1], [scenario 2].
metadata: {"clawdbot":{"emoji":"🔧","requires":{"anyBins":["tool-name"]}}}
---

# Tool Name

[One paragraph: what it does and why you'd use it.]

## When to Use
- [4-6 scenarios]

## Quick Reference
[Most common commands with examples]

## Common Operations
### [Operation 1]
### [Operation 2]

## Advanced Patterns
### [Pattern 1]

## Troubleshooting
### [Common error and fix]

## Tips
```

### Language/Framework Reference

For skills about patterns in a specific language or framework.

```markdown
---
name: pattern-name
description: [Pattern] in [language/framework]. Use when [scenario 1], [scenario 2].
metadata: {"clawdbot":{"emoji":"📐","requires":{"anyBins":["runtime"]}}}
---

# Pattern Name

## When to Use

## Quick Reference
[Cheat sheet / syntax summary]

## Patterns
### [Pattern 1 — with full example]
### [Pattern 2 — with full example]

## Cross-Language Comparison (if applicable)

## Anti-Patterns
[What NOT to do, with explanation]

## Tips
```

### Workflow/Process Guide

For skills about multi-step processes.

```markdown
---
name: workflow-name
description: [Workflow description]. Use when [scenario 1], [scenario 2].
metadata: {"clawdbot":{"emoji":"🔄","requires":{"anyBins":["tool1","tool2"]}}}
---

# Workflow Name

## When to Use

## Prerequisites
[What needs to be set up first]

## Step-by-Step
### Step 1: [Action]
### Step 2: [Action]
### Step 3: [Action]

## Variations
### [Variation for different context]

## Troubleshooting

## Tips
```

## Anti-Patterns

### Too abstract

```markdown
# BAD
## Error Handling
Error handling is important for robust applications. You should always
handle errors properly to prevent unexpected crashes...

# GOOD
## Error Handling
```bash
# Bash: exit on any error
set -euo pipefail

# Trap for cleanup on exit
trap 'rm -f "$TMPFILE"' EXIT
```
```

### Too narrow

```markdown
# BAD: Only useful for one specific case
---
name: react-useeffect-cleanup
description: How to clean up useEffect hooks in React
---

# GOOD: Broad enough to be a real reference
---
name: react-hooks
description: React hooks patterns. Use when working with useState, useEffect, useCallback, useMemo, custom hooks, or debugging hook-related issues.
---
```

### Wall of text without examples

If any section goes more than 10 lines without a code block, it's too text-heavy. Break it up with examples.

### Missing cross-references

If your skill mentions another tool or concept that has its own skill, note it:

```markdown
# For Docker networking issues, see the `container-debug` skill.
# For regex syntax details, see the `regex-patterns` skill.
```

### Outdated commands

Verify every command works on current tool versions. Common traps:
- Docker Compose: `docker-compose` (v1) vs. `docker compose` (v2)
- Python: `pip` vs. `pip3`, `python` vs. `python3`
- Node.js: CommonJS (`require`) vs. ESM (`import`)

## Size Guidelines

| Metric | Target | Too Short | Too Long |
|---|---|---|---|
| Total lines | 300-550 | < 150 | > 700 |
| Sections | 5-10 | < 3 | > 15 |
| Code blocks | 15-40 | < 8 | > 60 |
| Tips | 5-10 | < 3 | > 15 |

A skill under 150 lines probably lacks examples. A skill over 700 lines should be split into two skills.

## Publishing Checklist

Before publishing, verify:

1. **Frontmatter is valid YAML** — test by pasting into a YAML validator
2. **Description starts with what the skill does** — not "This skill..." or "A skill for..."
3. **Every section has at least one code block** — no text-only sections in the main content
4. **Commands actually work** — test in a clean environment
5. **No placeholder values left** — search for `TODO`, `FIXME`, `example.com` used as real URLs
6. **Slug is available** — `npx molthub@latest search "your-slug"` returns no exact match
7. **`requires.anyBins` lists real dependencies** — tools the skill's commands actually invoke
8. **Tips section exists** — with 5+ actionable, non-obvious bullets

## Publishing

```bash
# Publish a new skill
npx molthub@latest publish ./skills/my-skill \
  --slug my-skill \
  --name "My Skill" \
  --version 1.0.0 \
  --changelog "Initial release"

# Update an existing skill
npx molthub@latest publish ./skills/my-skill \
  --slug my-skill \
  --name "My Skill" \
  --version 1.1.0 \
  --changelog "Added new section on X"

# Verify it's published
npx molthub@latest search "my-skill"
```

## Tips

- The `description` field is your skill's search ranking. Spend more time on it than any single content section. Include the specific verbs and nouns users would search for.
- Lead with the most common use case. If 80% of users need "how to encode Base64", put that before "how to convert between MessagePack and CBOR."
- Every code example should be copy-pasteable. If it needs setup that isn't shown, add the setup.
- Write for the agent, not the human. The agent needs unambiguous instructions it can follow step by step. Avoid "you might want to consider" — say "do X when Y."
- Test your skill by asking an agent to use it on a real task. If the agent can't follow the instructions to produce a correct result, the skill needs work.
- Prefer `bash` code blocks for commands, even in language-specific skills. The agent often operates via shell, and bash blocks signal "run this."
- Don't duplicate what `--help` already provides. Focus on patterns, combinations, and the non-obvious things that `--help` doesn't teach.
- Version your skills semantically: patch for typo fixes, minor for new sections, major for restructures. The registry tracks version history.
