---
name: skill-security-auditor
description: Command-line security analyzer for ClawHub skills. Run analyze-skill.sh to scan SKILL.md files for malicious patterns, credential leaks, and C2 infrastructure before installation. Includes threat intelligence database with 20+ detection patterns.
emoji: 🔍🛡️
metadata:
  openclaw:
    requires:
      bins: ["curl", "jq", "grep", "bash"]
    version: "1.0.0"
    author: "akm626"
    category: "security"
    tags: ["security", "audit", "malware-detection", "skill-vetting", "cli-tool"]
---

# Skill Security Auditor

## Description

The Skill Security Auditor is a **command-line tool** that performs pattern-based security analysis of ClawHub skills before installation. Given the recent discovery of 341+ malicious skills (ClawHavoc campaign) that distributed Atomic Stealer (AMOS) and stole cryptocurrency credentials, this tool provides essential pre-installation threat detection.

**What this skill provides:**
- ✅ Bash script (`analyze-skill.sh`) for local security analysis
- ✅ Threat intelligence database (`patterns/malicious-patterns.json`)
- ✅ Pattern matching against 20+ known malicious indicators
- ✅ Risk scoring system (0-100 scale)
- ✅ Detailed audit reports with recommendations

**How to use it:**
1. Install this skill from ClawHub
2. Run the `analyze-skill.sh` script against any skill (by slug or local file)
3. Review the risk assessment and findings
4. Make informed decision about installation

**Use this tool when:**
- About to install a new skill from ClawHub
- Investigating suspicious skill behavior  
- Performing security due diligence on community skills
- Auditing your currently installed skills

**This tool does NOT:**
- ❌ Automatically scan skills (you run it manually)
- ❌ Block installations (it's advisory only)
- ❌ Access VirusTotal API (use ClawHub's web interface for that)
- ❌ Guarantee 100% detection (defense in depth recommended)

## Core Capabilities

### 1. **Malicious Pattern Detection**
Scans for known malicious patterns from the ClawHavoc campaign:
- Fake prerequisite installations (openclaw-agent.zip, openclaw-setup.exe)
- Suspicious download commands in SKILL.md
- Hidden payload execution in metadata
- Social engineering language patterns
- Unauthorized external binary downloads

### 2. **Credential Leak Analysis**
Identifies potential credential exposure vectors:
- Hardcoded API keys, tokens, passwords in SKILL.md
- Suspicious environment variable exfiltration
- Unencrypted sensitive data transmission
- Overly broad permission requests
- Credential harvesting patterns

### 3. **Dependency Chain Validation**
Analyzes skill dependencies for:
- Unverified binary requirements
- Suspicious GitHub repository sources
- External script execution
- Network connections to unknown hosts
- Nested dependency exploitation

### 4. **C2 Infrastructure Detection**
Checks for Command & Control indicators:
- Known malicious IP addresses (e.g., 91.92.242.30 from ClawHavoc)
- Suspicious domain patterns
- Encoded communication endpoints
- Data exfiltration channels
- Beaconing behavior patterns

### 5. **SKILL.md Structure Validation**
Verifies skill integrity:
- Proper YAML frontmatter structure
- Metadata consistency
- Description clarity vs actual behavior
- Permission justification
- Author verification (GitHub account age)

## Security Scoring System

Each analyzed skill receives a **Risk Score (0-100)**:

- **0-20**: ✅ **SAFE** - No significant security concerns
- **21-40**: ⚠️ **LOW RISK** - Minor concerns, proceed with caution
- **41-60**: 🟡 **MEDIUM RISK** - Multiple red flags, manual review recommended
- **61-80**: 🔴 **HIGH RISK** - Serious concerns, do NOT install without expert review
- **81-100**: ☠️ **CRITICAL** - Malicious indicators detected, AVOID installation

## Usage Instructions

This skill provides a **bash script** (`analyze-skill.sh`) that performs pattern-based security analysis of ClawHub skills. The analysis runs locally using the included threat intelligence database.

### Installation & Setup

```bash
# Install the skill from ClawHub
npx clawhub install skill-security-auditor

# Make the analyzer executable
chmod +x ~/.openclaw/skills/skill-security-auditor/analyze-skill.sh

# Optional: Create alias for convenience
echo 'alias audit-skill="~/.openclaw/skills/skill-security-auditor/analyze-skill.sh"' >> ~/.bashrc
source ~/.bashrc
```

### Audit a Skill Before Installing

**Method 1: Analyze by slug (automatic fetch from ClawHub)**

```bash
~/.openclaw/skills/skill-security-auditor/analyze-skill.sh --slug bitcoin-tracker

# Example output:
# ============================================
#          SECURITY AUDIT REPORT
# ============================================
# 
# Risk Score: 85/100 - ☠️ CRITICAL
# ...
```

**Method 2: Analyze local file**

```bash
# Download skill first
curl -s "https://clawhub.ai/api/skills/bitcoin-tracker/latest" > /tmp/skill.md

# Then analyze
~/.openclaw/skills/skill-security-auditor/analyze-skill.sh --file /tmp/skill.md
```

### Audit All Installed Skills

```bash
# Scan all skills in your workspace
for skill in ~/.openclaw/skills/*/SKILL.md; do
  echo "Checking: $(basename $(dirname $skill))"
  ~/.openclaw/skills/skill-security-auditor/analyze-skill.sh -f "$skill"
done
```

### Quick Manual Security Check

```bash
# Fast grep-based pattern matching (no full analysis)
grep -iE "(prerequisite.*download|91\.92\.242\.30|curl.*\|.*bash)" SKILL.md
```

## Detection Heuristics

### 🚨 CRITICAL Red Flags (Auto-fail)

1. **Fake Prerequisites Section**
   - Matches: "Prerequisites", "Setup Required", "Installation Steps"
   - Contains: Download links to `.zip`, `.exe`, `.dmg` files
   - Example: "Download openclaw-agent.zip from https://..."

2. **Known Malicious Infrastructure**
   - IP: `91.92.242.30` (ClawHavoc C2)
   - Domains: Newly registered or suspicious TLDs
   - Encoded URLs or base64 obfuscation

3. **Credential Harvesting**
   - Regex patterns for API keys: `(api[_-]?key|token|password)\s*[:=]\s*['\"][^'\"]+['\"]`
   - SSH key access requests
   - Wallet private key patterns

4. **Unauthorized Code Execution**
   - `curl | bash` or `wget | sh` patterns
   - Hidden base64 encoded commands
   - Dynamic eval() or exec() on external input

### ⚠️ Warning Indicators (Score increase)

1. **Suspicious Dependencies**
   - Binary requirements without clear justification
   - Dependencies from unverified sources
   - Excessive permission requests

2. **Obfuscation Techniques**
   - Heavily encoded strings in metadata
   - Minified or obfuscated JavaScript/Python
   - Redirect chains in URLs

3. **Social Engineering Language**
   - Urgency phrases: "Install immediately", "Limited time"
   - Authority claims: "Official OpenClaw", "Verified by Anthropic"
   - Fear tactics: "Your system is at risk without this"

### ✅ Positive Security Indicators

1. **Verified Author**
   - GitHub account > 1 year old
   - Multiple well-rated skills
   - Active community engagement

2. **Transparent Dependencies**
   - Clear binary requirements with official sources
   - Open-source tool dependencies
   - Well-documented permission needs

3. **Code Quality**
   - Clean, readable SKILL.md
   - Proper error handling
   - No unnecessary network calls

## Audit Report Format

```markdown
## Security Audit Report
**Skill**: {skill-name}
**Author**: {author}
**Version**: {version}
**Audit Date**: {date}

### Risk Score: {score}/100 - {RISK_LEVEL}

### Critical Findings:
- {finding 1}
- {finding 2}

### Warning Indicators:
- {warning 1}
- {warning 2}

### Positive Indicators:
- {positive 1}
- {positive 2}

### Recommendations:
{INSTALL | DO NOT INSTALL | REVIEW MANUALLY}

### Detailed Analysis:
{Deep dive into specific concerns}

### VirusTotal Link:
{If available from ClawHub}
```

## Integration with VirusTotal

**Important**: This skill does NOT directly access VirusTotal's API. Instead, VirusTotal integration is available through ClawHub's web interface via their partnership with VirusTotal.

To check VirusTotal results for a skill:

1. Visit the skill's ClawHub page: `https://clawhub.ai/skills/{skill-slug}`
2. Look for the VirusTotal scan results on the skill's page
3. ClawHub automatically scans published skills via their VirusTotal partnership

**This analyzer focuses on pattern-based threat detection.** It complements (but does not replace) ClawHub's VirusTotal scanning.

### Recommended Security Workflow

1. **Run this analyzer first** - Pattern-based detection (local, instant)
2. **Check ClawHub's VirusTotal results** - Binary/file reputation (if available)
3. **Manual code review** - Final verification for critical use cases

```bash
# Step 1: Pattern analysis (local)
~/.openclaw/skills/skill-security-auditor/analyze-skill.sh -s suspicious-skill

# Step 2: Visit ClawHub page for VirusTotal results
# https://clawhub.ai/skills/suspicious-skill

# Step 3: Manual review if needed
curl -s "https://clawhub.ai/api/skills/suspicious-skill/latest" > skill.md
less skill.md
```

## Example Workflow

**Scenario**: User wants to install a skill called `solana-wallet-tracker`

**Step 1: Run Security Analysis**
```bash
$ ~/.openclaw/skills/skill-security-auditor/analyze-skill.sh -s solana-wallet-tracker

Fetching skill 'solana-wallet-tracker' from ClawHub...
✓ Skill fetched successfully

Analyzing skill content...

============================================
         SECURITY AUDIT REPORT
============================================

Risk Score: 95/100 - ☠️ CRITICAL

============================================

☠️ CRITICAL FINDINGS:
  CLAW-001: Fake Prerequisites - ClawHavoc Campaign [+50 points]
  └─ Matches the ClawHavoc campaign pattern of fake prerequisites requesting malicious binary downloads
  CLAW-002: Known C2 Infrastructure [+50 points]
  └─ IP address used in ClawHavoc campaign for C2 communications

============================================
RECOMMENDATION:
DO NOT INSTALL. Malicious patterns detected matching known attack campaigns.
============================================
```

**Step 2: Decision**
- ☠️ **CRITICAL Risk** → **DO NOT INSTALL**
- Report skill to ClawHub moderators
- Look for safe alternatives

**Step 3: Verify on ClawHub** (optional)
```bash
# Visit skill page to check VirusTotal results
open "https://clawhub.ai/skills/solana-wallet-tracker"
```

## Advanced Features

### 1. Behavioral Analysis (Future Enhancement)
- Sandbox execution monitoring
- Network traffic analysis
- File system access patterns

### 2. Community Threat Intelligence
- Share malicious skill signatures
- Collaborative IOC database
- Reputation scoring system

### 3. Continuous Monitoring
- Auto-audit skills on updates
- Alert on new security advisories
- Periodic re-scanning of installed skills

## False Positive Mitigation

To minimize false positives:

1. **Contextual Analysis**: Binary requirements for legitimate tools (e.g., `gh` for GitHub CLI) are validated against known safe sources
2. **Whitelisting**: Verified authors and established skills get trust bonuses
3. **Human Review Option**: Always provide detailed reasoning for security decisions
4. **Appeal Process**: Users can report false positives for skill reputation adjustment

## Compliance & Ethics

This skill:
- ✅ Analyzes publicly available skill metadata
- ✅ Protects user security and privacy
- ✅ Promotes responsible skill development
- ❌ Does NOT perform unauthorized access
- ❌ Does NOT guarantee 100% security (nothing does)
- ❌ Does NOT replace user judgment

## Response Templates

### Safe Skill
```
✅ Security Audit Complete

{skill-name} has been analyzed and appears SAFE to install.

Risk Score: {score}/100 (LOW)

No malicious patterns detected. The skill:
- Uses standard dependencies from trusted sources
- Has a verified author with {X} published skills
- Contains clear documentation with no obfuscation
- Requests appropriate permissions for its function

VirusTotal: {link}

Recommendation: Safe to proceed with installation.
```

### Suspicious Skill
```
🔴 Security Alert: HIGH RISK DETECTED

{skill-name} has been flagged with CRITICAL security concerns.

Risk Score: {score}/100 (HIGH)

⚠️ Critical Findings:
{detailed findings}

This skill matches patterns from the ClawHavoc malware campaign.

Recommendation: DO NOT INSTALL. Consider reporting this skill to ClawHub moderators.

Alternative safe skills: {suggestions}
```

## Technical Implementation Notes

**Pattern Database Location**: `~/.openclaw/security-auditor/patterns/`
- `malicious-patterns.json`: Known bad indicators
- `safe-patterns.json`: Whitelisted elements
- `ioc-database.json`: Indicators of Compromise

**Update Mechanism**:
```bash
# Pull latest threat intelligence
curl -s "https://openclaw-security.github.io/threat-intel/latest.json" \
  > ~/.openclaw/security-auditor/patterns/ioc-database.json
```

## Contributing

Found a new malicious pattern? Submit IOCs to the OpenClaw Security Working Group:
- GitHub: github.com/openclaw/security-auditor
- Report Format: JSON with pattern regex, description, severity

## Limitations

⚠️ **Important Disclaimers**:
- This tool provides analysis, not guarantees
- Sophisticated malware may evade detection
- Always combine with VirusTotal + manual review for critical applications
- Security is a shared responsibility
- No automated tool replaces security expertise

## References

- ClawHavoc Campaign Analysis: [The Hacker News, Feb 2026]
- OpenClaw Security Partnership: VirusTotal Integration
- Malicious Skill Database: OpenSourceMalware Research
- ClawHub Moderation Guide: docs.openclaw.ai/security

---

**Remember**: The best security is defense in depth. Use this skill as ONE layer of your security strategy, not the only layer.

Stay safe, stay skeptical, stay secure. 🦞🛡️
