#!/bin/bash
# publish.sh - Complete publish workflow to GitHub and ClawHub
# Usage: ./publish.sh <skill-path> [options]

set -e

SKILL_PATH="${1:-}"
GITHUB_REPO=""
CLAWHUB_SLUG=""
VERSION="1.0.0"
SKIP_GITHUB=false
SKIP_CLAWHUB=false

# Parse options
shift || true
while [[ $# -gt 0 ]]; do
    case $1 in
        --github)
            GITHUB_REPO="$2"
            shift 2
            ;;
        --clawhub-slug)
            CLAWHUB_SLUG="$2"
            shift 2
            ;;
        --version)
            VERSION="$2"
            shift 2
            ;;
        --skip-github)
            SKIP_GITHUB=true
            shift
            ;;
        --skip-clawhub)
            SKIP_CLAWHUB=true
            shift
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Validate inputs
if [ -z "$SKILL_PATH" ]; then
    echo "Usage: $0 <skill-path> [options]"
    echo ""
    echo "Options:"
    echo "  --github REPO       GitHub repo (owner/repo)"
    echo "  --clawhub-slug      ClawHub slug"
    echo "  --version VER       Version (default: 1.0.0)"
    echo "  --skip-github       Skip GitHub push"
    echo "  --skip-clawhub      Skip ClawHub publish"
    exit 1
fi

if [ ! -d "$SKILL_PATH" ]; then
    echo "Error: Skill path not found: $SKILL_PATH"
    exit 1
fi

SKILL_NAME=$(basename "$SKILL_PATH")

echo "Publishing: $SKILL_NAME"
echo "Version: $VERSION"
echo ""

# Check for required tools
if ! $SKIP_GITHUB && [ -n "$GITHUB_REPO" ]; then
    if ! command -v git &> /dev/null; then
        echo "Error: git not found"
        exit 1
    fi
fi

if ! $SKIP_CLAWHUB && [ -n "$CLAWHUB_SLUG" ]; then
    if ! command -v clawhub &> /dev/null; then
        echo "Error: clawhub CLI not found. Install with: npm i -g clawhub"
        exit 1
    fi
fi

# Step 1: Validate skill
echo "=== Step 1: Validating Skill ==="
if [ ! -f "$SKILL_PATH/SKILL.md" ]; then
    echo "Error: SKILL.md not found"
    exit 1
fi

echo "✓ SKILL.md exists"

# Check frontmatter
if ! grep -q "^---" "$SKILL_PATH/SKILL.md"; then
    echo "Warning: No YAML frontmatter found in SKILL.md"
fi

echo "✓ Validation passed"
echo ""

# Step 2: GitHub Push
if ! $SKIP_GITHUB && [ -n "$GITHUB_REPO" ]; then
    echo "=== Step 2: Pushing to GitHub ==="
    
    cd "$SKILL_PATH"
    
    # Initialize git if needed
    if [ ! -d ".git" ]; then
        git init
        echo "✓ Git initialized"
    fi
    
    # Add remote if not exists
    if ! git remote | grep -q origin; then
        git remote add origin "git@github.com:$GITHUB_REPO.git"
        echo "✓ Remote added"
    fi
    
    # Stage all files
    git add -A
    
    # Commit
    if git diff --cached --quiet; then
        echo "No changes to commit"
    else
        git commit -m "Release $SKILL_NAME v$VERSION"
        echo "✓ Changes committed"
    fi
    
    # Tag
    git tag -a "v$VERSION" -m "Release $SKILL_NAME v$VERSION" 2>/dev/null || echo "Tag v$VERSION already exists"
    
    # Push
    echo "Pushing to GitHub..."
    git push origin main 2>/dev/null || git push origin master 2>/dev/null || git push origin HEAD 2>/dev/null || echo "Push failed - may need manual intervention"
    git push origin --tags 2>/dev/null || true
    
    echo "✓ GitHub push complete"
    echo ""
fi

# Step 3: ClawHub Publish
if ! $SKIP_CLAWHUB && [ -n "$CLAWHUB_SLUG" ]; then
    echo "=== Step 3: Publishing to ClawHub ==="
    
    # Check login
    if ! clawhub whoami &> /dev/null; then
        echo "Error: Not logged in to ClawHub. Run: clawhub login"
        exit 1
    fi
    
    echo "Publishing $SKILL_NAME to ClawHub..."
    clawhub publish "$SKILL_PATH" \
        --slug "$CLAWHUB_SLUG" \
        --name "$SKILL_NAME" \
        --version "$VERSION" \
        --changelog "Release v$VERSION"
    
    echo "✓ ClawHub publish complete"
    echo ""
fi

# Summary
echo "=== Publish Summary ==="
echo "Skill: $SKILL_NAME"
echo "Version: $VERSION"

if ! $SKIP_GITHUB && [ -n "$GITHUB_REPO" ]; then
    echo "GitHub: https://github.com/$GITHUB_REPO"
fi

if ! $SKIP_CLAWHUB && [ -n "$CLAWHUB_SLUG" ]; then
    echo "ClawHub: https://clawhub.com/s/$CLAWHUB_SLUG"
fi

echo ""
echo "✓ Publish workflow complete!"
