#!/bin/bash
# extract-from-memory.sh - Extract skill patterns from memory markdown
# Usage: ./extract-from-memory.sh <memory.md> <output-dir>

set -e

MEMORY_FILE="${1:-}"
OUTPUT_DIR="${2:-}"

# Validate inputs
if [ -z "$MEMORY_FILE" ] || [ -z "$OUTPUT_DIR" ]; then
    echo "Usage: $0 <memory.md> <output-dir>"
    exit 1
fi

if [ ! -f "$MEMORY_FILE" ]; then
    echo "Error: Memory file not found: $MEMORY_FILE"
    exit 1
fi

mkdir -p "$OUTPUT_DIR"

echo "Extracting from: $MEMORY_FILE"

# Copy memory file
cp "$MEMORY_FILE" "$OUTPUT_DIR/source-memory.md"

# Extract TODO items
echo ""
echo "=== Extracting TODOs/Tasks ==="
grep -E "^- \[([ x])\]" "$MEMORY_FILE" > "$OUTPUT_DIR/todos.txt" 2>/dev/null || true

if [ -s "$OUTPUT_DIR/todos.txt" ]; then
    echo "TODOs found:"
    cat "$OUTPUT_DIR/todos.txt" | head -10
else
    echo "No TODOs found"
fi

# Extract completed tasks
echo ""
echo "=== Completed Work ==="
grep -E "^- \[x\]" "$MEMORY_FILE" > "$OUTPUT_DIR/completed.txt" 2>/dev/null || true

if [ -s "$OUTPUT_DIR/completed.txt" ]; then
    echo "Completed tasks found:"
    cat "$OUTPUT_DIR/completed.txt" | head -10
fi

# Extract decisions
echo ""
echo "=== Key Decisions ==="
grep -E "^(##|###).*[Dd]ecision|决定|选择" "$MEMORY_FILE" > "$OUTPUT_DIR/decisions.txt" 2>/dev/null || true

# Extract code blocks
echo ""
echo "=== Code Blocks ==="
awk '/^\`\`\`/{p=!p;next}p' "$MEMORY_FILE" > "$OUTPUT_DIR/code_blocks.txt" 2>/dev/null || true

if [ -s "$OUTPUT_DIR/code_blocks.txt" ]; then
    echo "Code blocks found: $(wc -l < "$OUTPUT_DIR/code_blocks.txt") lines"
fi

# Generate summary
cat > "$OUTPUT_DIR/extraction-summary.md" << EOF
# Memory Extraction Summary

## Source
- File: $(basename "$MEMORY_FILE")
- Date: $(date)

## Extracted Content
EOF

[ -s "$OUTPUT_DIR/todos.txt" ] && echo "- TODOs: $(wc -l < "$OUTPUT_DIR/todos.txt") items" >> "$OUTPUT_DIR/extraction-summary.md"
[ -s "$OUTPUT_DIR/completed.txt" ] && echo "- Completed: $(wc -l < "$OUTPUT_DIR/completed.txt") items" >> "$OUTPUT_DIR/extraction-summary.md"
[ -s "$OUTPUT_DIR/code_blocks.txt" ] && echo "- Code blocks: $(wc -l < "$OUTPUT_DIR/code_blocks.txt") lines" >> "$OUTPUT_DIR/extraction-summary.md"

cat >> "$OUTPUT_DIR/extraction-summary.md" << EOF

## Skill Ideas
Review completed.txt for workflows that could be automated.
Review code_blocks.txt for reusable scripts.

## Next Steps
1. Review extracted content
2. Identify reusable patterns
3. Run create-skill.sh to package as skill
EOF

echo ""
echo "✓ Extraction complete"
echo "  Review: $OUTPUT_DIR/extraction-summary.md"
