#!/bin/bash
# extract-from-history.sh - Extract skill patterns from conversation history
# Usage: ./extract-from-history.sh <session.jsonl> <output-dir> [options]

set -e

SESSION_FILE="${1:-}"
OUTPUT_DIR="${2:-}"
SINCE_DATE=""
PATTERN=""
TOOLS_ONLY=false

# Parse options
shift 2 || true
while [[ $# -gt 0 ]]; do
    case $1 in
        --since)
            SINCE_DATE="$2"
            shift 2
            ;;
        --pattern)
            PATTERN="$2"
            shift 2
            ;;
        --tools-only)
            TOOLS_ONLY=true
            shift
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Validate inputs
if [ -z "$SESSION_FILE" ] || [ -z "$OUTPUT_DIR" ]; then
    echo "Usage: $0 <session.jsonl> <output-dir> [options]"
    echo ""
    echo "Options:"
    echo "  --since DATE     Only extract from DATE onwards (YYYY-MM-DD)"
    echo "  --pattern REGEX  Filter messages matching pattern"
    echo "  --tools-only     Only extract tool usage patterns"
    exit 1
fi

if [ ! -f "$SESSION_FILE" ]; then
    echo "Error: Session file not found: $SESSION_FILE"
    exit 1
fi

mkdir -p "$OUTPUT_DIR"

echo "Extracting from: $SESSION_FILE"
echo "Output to: $OUTPUT_DIR"

# Extract user requests and assistant responses
echo ""
echo "=== Extracting Conversation Flow ==="

# Parse JSONL and extract relevant content
jq -r '
    select(.type == "message") |
    select(.message.role == "user" or .message.role == "assistant") |
    {
        role: .message.role,
        content: (.message.content // [] | 
            map(select(.type == "text") | .text) | 
            join("\n")
        ),
        timestamp: .timestamp
    } | 
    "[" + .role + "] " + .content
' "$SESSION_FILE" 2>/dev/null | head -100 > "$OUTPUT_DIR/conversation.txt" || echo "Note: jq not available, using grep fallback"

# Fallback if jq not available
if [ ! -s "$OUTPUT_DIR/conversation.txt" ]; then
    grep -o '"text":"[^"]*"' "$SESSION_FILE" | sed 's/"text":"//;s/"$//' | head -100 > "$OUTPUT_DIR/conversation.txt"
fi

echo "✓ Conversation extracted ($(wc -l < "$OUTPUT_DIR/conversation.txt") lines)"

# Extract tool calls if present
if [ -s "$OUTPUT_DIR/conversation.txt" ]; then
    echo ""
    echo "=== Identified Patterns ==="
    
    # Look for common skill patterns
    grep -E "(脚本|script|备份|backup|定时|cron|任务|task|自动化|automate)" "$OUTPUT_DIR/conversation.txt" | head -20 > "$OUTPUT_DIR/patterns.txt" || true
    
    if [ -s "$OUTPUT_DIR/patterns.txt" ]; then
        echo "Potential skill patterns found:"
        cat "$OUTPUT_DIR/patterns.txt" | head -10
    fi
fi

# Extract code blocks
echo ""
echo "=== Extracting Code/Scripts ==="
grep -E "(bash|python|#!/bin)" "$OUTPUT_DIR/conversation.txt" | head -10 > "$OUTPUT_DIR/code_hints.txt" || true

if [ -s "$OUTPUT_DIR/code_hints.txt" ]; then
    echo "Code/script patterns found"
fi

# Generate summary
echo ""
echo "=== Summary ==="
cat > "$OUTPUT_DIR/extraction-summary.md" << EOF
# Extraction Summary

## Source
- Session: $(basename "$SESSION_FILE")
- Date: $(date)

## Extracted Files
- conversation.txt - Full conversation text
- patterns.txt - Identified skill patterns
- code_hints.txt - Code/script references

## Next Steps
1. Review patterns.txt for skill ideas
2. Check code_hints.txt for reusable scripts
3. Run create-skill.sh to generate skill structure

## Suggested Skill Name
$(basename "$OUTPUT_DIR")

## Suggested Description
$(head -1 "$OUTPUT_DIR/patterns.txt" 2>/dev/null || echo "Based on conversation from $(basename "$SESSION_FILE")")
EOF

echo "✓ Extraction complete"
echo "  Review: $OUTPUT_DIR/extraction-summary.md"
