#!/bin/bash
# create-skill.sh - Create skill structure from extracted content
# Usage: ./create-skill.sh <extracted-dir> <skill-name> [options]

set -e

EXTRACTED_DIR="${1:-}"
SKILL_NAME="${2:-}"
DESCRIPTION=""
SKILL_TYPE="workflow"

# Parse options
shift 2 || true
while [[ $# -gt 0 ]]; do
    case $1 in
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --type)
            SKILL_TYPE="$2"
            shift 2
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Validate inputs
if [ -z "$EXTRACTED_DIR" ] || [ -z "$SKILL_NAME" ]; then
    echo "Usage: $0 <extracted-dir> <skill-name> [options]"
    echo ""
    echo "Options:"
    echo "  --description TEXT  Skill description"
    echo "  --type TYPE         Skill type (workflow|tool|reference)"
    exit 1
fi

if [ ! -d "$EXTRACTED_DIR" ]; then
    echo "Error: Extracted directory not found: $EXTRACTED_DIR"
    exit 1
fi

# Create skill directory
SKILL_DIR="./$SKILL_NAME"
mkdir -p "$SKILL_DIR/scripts"

echo "Creating skill: $SKILL_NAME"
echo "Type: $SKILL_TYPE"

# Generate description if not provided
if [ -z "$DESCRIPTION" ]; then
    if [ -f "$EXTRACTED_DIR/extraction-summary.md" ]; then
        DESCRIPTION=$(grep "Suggested Description" "$EXTRACTED_DIR/extraction-summary.md" | cut -d':' -f2- | xargs || echo "")
    fi
    if [ -z "$DESCRIPTION" ]; then
        DESCRIPTION="Skill created from $(basename "$EXTRACTED_DIR")"
    fi
fi

# Create SKILL.md
cat > "$SKILL_DIR/SKILL.md" << EOF
---
name: $SKILL_NAME
description: $DESCRIPTION
---

# $(echo "$SKILL_NAME" | sed 's/-/ /g' | awk '{for(i=1;i<=NF;i++) $i=toupper(substr($i,1,1)) substr($i,2)} 1')

## Overview

${DESCRIPTION}

## Quick Start

\`\`\`bash
# Add usage example here
./scripts/main.sh
\`\`\`

## Usage

[Add detailed usage instructions based on extracted content]

## Scripts

- \`scripts/main.sh\` - Main script

EOF

# Copy any extracted code blocks as starter scripts
if [ -f "$EXTRACTED_DIR/code_blocks.txt" ] && [ -s "$EXTRACTED_DIR/code_blocks.txt" ]; then
    echo ""
    echo "=== Including Extracted Code ==="
    cp "$EXTRACTED_DIR/code_blocks.txt" "$SKILL_DIR/scripts/extracted-code.sh"
    chmod +x "$SKILL_DIR/scripts/extracted-code.sh" 2>/dev/null || true
    echo "✓ Starter script created from extracted code"
fi

# Add scripts section to SKILL.md based on type
case "$SKILL_TYPE" in
    workflow)
        cat >> "$SKILL_DIR/SKILL.md" << EOF

## Workflow

1. Step one
2. Step two
3. Step three

EOF
        ;;
    tool)
        cat >> "$SKILL_DIR/SKILL.md" << EOF

## Available Tools

### Tool 1
Description and usage

### Tool 2
Description and usage

EOF
        ;;
    reference)
        cat >> "$SKILL_DIR/SKILL.md" << EOF

## Reference

Key information and guidelines

EOF
        ;;
esac

# Create placeholder main script
cat > "$SKILL_DIR/scripts/main.sh" << 'EOF'
#!/bin/bash
# Main script for skill

set -e

echo "Skill: $(basename $(dirname $0))"
echo "Add your implementation here"
EOF

chmod +x "$SKILL_DIR/scripts/main.sh"

# Create README for the skill
cat > "$SKILL_DIR/README.md" << EOF
# $SKILL_NAME

$DESCRIPTION

## Installation

\`\`\`bash
clawhub install $SKILL_NAME
\`\`\`

## Usage

See SKILL.md for detailed usage instructions.

## Files

- SKILL.md - Skill definition
- scripts/ - Executable scripts

---

Created from: $(basename "$EXTRACTED_DIR")
Date: $(date)
EOF

echo ""
echo "✓ Skill structure created: $SKILL_DIR"
echo ""
echo "Next steps:"
echo "1. Edit $SKILL_DIR/SKILL.md with complete instructions"
echo "2. Implement scripts in $SKILL_DIR/scripts/"
echo "3. Test the skill locally"
echo "4. Run publish.sh to publish"
