#!/bin/bash
# create-and-publish.sh - One command to extract, create, and publish skill
# Usage: ./create-and-publish.sh --source <path> --skill-name <name> [options]

set -e

SOURCE=""
SOURCE_TYPE="auto"
SKILL_NAME=""
GITHUB_REPO=""
CLAWHUB_SLUG=""
VERSION="1.0.0"
DESCRIPTION=""

# Show usage
show_usage() {
    echo "Usage: $0 --source <path> --skill-name <name> [options]"
    echo ""
    echo "Required:"
    echo "  --source PATH       Source file (session.jsonl or memory.md)"
    echo "  --skill-name NAME   Name for the new skill"
    echo ""
    echo "Options:"
    echo "  --source-type TYPE  Source type (jsonl|md|auto)"
    echo "  --github REPO       GitHub repo (owner/repo)"
    echo "  --clawhub-slug      ClawHub slug"
    echo "  --version VER       Version (default: 1.0.0)"
    echo "  --description TEXT  Skill description"
    echo ""
    echo "Example:"
    echo "  $0 --source ~/.openclaw/agents/main/sessions/latest.jsonl \\"
    echo "     --skill-name my-automation \\"
    echo "     --github user/skills \\"
    echo "     --clawhub-slug my-automation"
}

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --source)
            SOURCE="$2"
            shift 2
            ;;
        --source-type)
            SOURCE_TYPE="$2"
            shift 2
            ;;
        --skill-name)
            SKILL_NAME="$2"
            shift 2
            ;;
        --github)
            GITHUB_REPO="$2"
            shift 2
            ;;
        --clawhub-slug)
            CLAWHUB_SLUG="$2"
            shift 2
            ;;
        --version)
            VERSION="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --help|-h)
            show_usage
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_usage
            exit 1
            ;;
    esac
done

# Validate required arguments
if [ -z "$SOURCE" ] || [ -z "$SKILL_NAME" ]; then
    echo "Error: --source and --skill-name are required"
    show_usage
    exit 1
fi

if [ ! -f "$SOURCE" ]; then
    echo "Error: Source file not found: $SOURCE"
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
EXTRACT_DIR="./extracted-$(date +%s)"

echo "=========================================="
echo "Skill Creation and Publish Pipeline"
echo "=========================================="
echo "Skill: $SKILL_NAME"
echo "Source: $SOURCE"
echo "Version: $VERSION"
echo ""

# Step 1: Extract
if [ "$SOURCE_TYPE" = "auto" ]; then
    if [[ "$SOURCE" == *.jsonl ]]; then
        SOURCE_TYPE="jsonl"
    elif [[ "$SOURCE" == *.md ]]; then
        SOURCE_TYPE="md"
    else
        echo "Error: Cannot auto-detect source type. Use --source-type"
        exit 1
    fi
fi

echo "=== Step 1: Extracting from $SOURCE_TYPE ==="

mkdir -p "$EXTRACT_DIR"

case "$SOURCE_TYPE" in
    jsonl)
        "$SCRIPT_DIR/extract-from-history.sh" "$SOURCE" "$EXTRACT_DIR"
        ;;
    md)
        "$SCRIPT_DIR/extract-from-memory.sh" "$SOURCE" "$EXTRACT_DIR"
        ;;
    *)
        echo "Error: Unknown source type: $SOURCE_TYPE"
        exit 1
        ;;
esac

# Step 2: Create Skill
echo ""
echo "=== Step 2: Creating Skill Structure ==="

SKILL_DIR="./$SKILL_NAME"
"$SCRIPT_DIR/create-skill.sh" "$EXTRACT_DIR" "$SKILL_NAME" \
    --description "$DESCRIPTION" \
    --type workflow

# Step 3: Publish
echo ""
echo "=== Step 3: Publishing ==="

PUBLISH_OPTS=""
[ -n "$GITHUB_REPO" ] && PUBLISH_OPTS="$PUBLISH_OPTS --github $GITHUB_REPO"
[ -n "$CLAWHUB_SLUG" ] && PUBLISH_OPTS="$PUBLISH_OPTS --clawhub-slug $CLAWHUB_SLUG"

if [ -z "$GITHUB_REPO" ] && [ -z "$CLAWHUB_SLUG" ]; then
    echo "Warning: No publish targets specified (--github or --clawhub-slug)"
    echo "Skill created but not published"
else
    "$SCRIPT_DIR/publish.sh" "$SKILL_DIR" \
        --version "$VERSION" \
        $PUBLISH_OPTS
fi

# Cleanup
echo ""
echo "=== Cleanup ==="
rm -rf "$EXTRACT_DIR"
echo "✓ Removed temporary files"

# Summary
echo ""
echo "=========================================="
echo "Pipeline Complete!"
echo "=========================================="
echo "Skill: $SKILL_NAME"
echo "Location: $SKILL_DIR"

if [ -n "$GITHUB_REPO" ]; then
    echo "GitHub: https://github.com/$GITHUB_REPO/tree/main/$SKILL_NAME"
fi

if [ -n "$CLAWHUB_SLUG" ]; then
    echo "ClawHub: clawhub install $CLAWHUB_SLUG"
fi

echo ""
echo "To use this skill:"
echo "  clawhub install $SKILL_NAME"
