#!/usr/bin/env python3
"""
Skill Exporter - Transform Clawdbot skills into standalone microservices.

Usage:
    python3 export.py --skill ~/.clawdbot/skills/instagram --target railway --llm anthropic
"""

import argparse
import os
import re
import shutil
import sys
from pathlib import Path

# Templates stored as strings for portability
TEMPLATES = {
    "api.py": '''#!/usr/bin/env python3
"""
{skill_name} API - Auto-generated by skill-exporter
"""
from fastapi import FastAPI, HTTPException, BackgroundTasks
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from typing import Optional
import os
import subprocess
import tempfile
from pathlib import Path
from dotenv import load_dotenv

load_dotenv()

app = FastAPI(
    title="{skill_name} API",
    description="{skill_description}",
    version="1.0.0"
)

app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Health check
@app.get("/health")
async def health():
    return {{"status": "ok", "service": "{skill_name}"}}

# Import LLM client if available
try:
    from llm_client import generate_text
    LLM_AVAILABLE = True
except ImportError:
    LLM_AVAILABLE = False
    def generate_text(prompt: str) -> str:
        raise HTTPException(503, "LLM not configured")

{endpoints}
''',

    "llm_client_anthropic.py": '''#!/usr/bin/env python3
"""
Anthropic LLM Client - Auto-generated by skill-exporter
"""
import os
import anthropic
from functools import lru_cache

@lru_cache()
def get_client():
    api_key = os.getenv("ANTHROPIC_API_KEY")
    if not api_key:
        raise ValueError("ANTHROPIC_API_KEY not set")
    return anthropic.Anthropic(api_key=api_key)

def generate_text(
    prompt: str,
    system: str = "You are a helpful assistant.",
    max_tokens: int = 1024,
    model: str = "claude-sonnet-4-20250514"
) -> str:
    """Generate text using Claude."""
    client = get_client()
    message = client.messages.create(
        model=model,
        max_tokens=max_tokens,
        system=system,
        messages=[{"role": "user", "content": prompt}]
    )
    return message.content[0].text

def generate_caption(topic: str, style: str = "engaging") -> str:
    """Generate a social media caption."""
    return generate_text(
        f"Write a {style} social media caption about: {topic}. Keep it under 200 characters. No hashtags yet.",
        system="You write concise, engaging social media captions."
    )

def generate_hashtags(topic: str, count: int = 5) -> list[str]:
    """Generate relevant hashtags."""
    result = generate_text(
        f"Generate {count} relevant hashtags for: {topic}. Return only hashtags, one per line.",
        system="You generate relevant social media hashtags."
    )
    return [tag.strip() for tag in result.strip().split("\\n") if tag.strip().startswith("#")]
''',

    "llm_client_openai.py": '''#!/usr/bin/env python3
"""
OpenAI LLM Client - Auto-generated by skill-exporter
"""
import os
from openai import OpenAI
from functools import lru_cache

@lru_cache()
def get_client():
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        raise ValueError("OPENAI_API_KEY not set")
    return OpenAI(api_key=api_key)

def generate_text(
    prompt: str,
    system: str = "You are a helpful assistant.",
    max_tokens: int = 1024,
    model: str = "gpt-4o"
) -> str:
    """Generate text using GPT."""
    client = get_client()
    response = client.chat.completions.create(
        model=model,
        max_tokens=max_tokens,
        messages=[
            {"role": "system", "content": system},
            {"role": "user", "content": prompt}
        ]
    )
    return response.choices[0].message.content

def generate_caption(topic: str, style: str = "engaging") -> str:
    """Generate a social media caption."""
    return generate_text(
        f"Write a {style} social media caption about: {topic}. Keep it under 200 characters. No hashtags yet.",
        system="You write concise, engaging social media captions."
    )

def generate_hashtags(topic: str, count: int = 5) -> list[str]:
    """Generate relevant hashtags."""
    result = generate_text(
        f"Generate {count} relevant hashtags for: {topic}. Return only hashtags, one per line.",
        system="You generate relevant social media hashtags."
    )
    return [tag.strip() for tag in result.strip().split("\\n") if tag.strip().startswith("#")]
''',

    "Dockerfile": '''FROM python:3.11-slim

WORKDIR /app

# Install system dependencies
{system_deps}

# Install Python dependencies
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copy application
COPY . .

# Make scripts executable
RUN chmod +x scripts/*.py 2>/dev/null || true

EXPOSE {port}

CMD ["uvicorn", "api:app", "--host", "0.0.0.0", "--port", "{port}"]
''',

    "docker-compose.yml": '''version: '3.8'

services:
  {skill_name}:
    build: .
    ports:
      - "{port}:{port}"
    env_file:
      - .env
    restart: unless-stopped
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:{port}/health"]
      interval: 30s
      timeout: 10s
      retries: 3
''',

    "requirements.txt": '''fastapi>=0.109.0
uvicorn[standard]>=0.27.0
python-dotenv>=1.0.0
requests>=2.31.0
pydantic>=2.5.0
{llm_deps}
''',

    "railway.json": '''{
  "$schema": "https://railway.app/railway.schema.json",
  "build": {
    "builder": "DOCKERFILE",
    "dockerfilePath": "Dockerfile"
  },
  "deploy": {
    "numReplicas": 1,
    "healthcheckPath": "/health",
    "healthcheckTimeout": 30,
    "restartPolicyType": "ON_FAILURE",
    "restartPolicyMaxRetries": 3
  }
}
''',

    "fly.toml": '''app = "{skill_name}-service"
primary_region = "fra"

[build]
  dockerfile = "Dockerfile"

[http_service]
  internal_port = {port}
  force_https = true
  auto_stop_machines = true
  auto_start_machines = true
  min_machines_running = 0

[[services]]
  protocol = "tcp"
  internal_port = {port}

  [[services.ports]]
    port = 80
    handlers = ["http"]

  [[services.ports]]
    port = 443
    handlers = ["tls", "http"]

  [[services.http_checks]]
    interval = "30s"
    timeout = "5s"
    path = "/health"
''',

    ".env.example": '''# {skill_name} Service Configuration
# Copy to .env and fill in values

{env_vars}
'''
}


def parse_skill_md(skill_path: Path) -> dict:
    """Parse SKILL.md frontmatter and extract metadata."""
    skill_md = skill_path / "SKILL.md"
    if not skill_md.exists():
        raise FileNotFoundError(f"No SKILL.md found in {skill_path}")
    
    content = skill_md.read_text()
    
    # Extract frontmatter
    frontmatter_match = re.match(r'^---\n(.*?)\n---', content, re.DOTALL)
    if not frontmatter_match:
        raise ValueError("Invalid SKILL.md: no frontmatter found")
    
    frontmatter = frontmatter_match.group(1)
    
    # Parse name and description
    name_match = re.search(r'^name:\s*(.+)$', frontmatter, re.MULTILINE)
    desc_match = re.search(r'^description:\s*(.+)$', frontmatter, re.MULTILINE)
    
    name = name_match.group(1).strip() if name_match else skill_path.name
    description = desc_match.group(1).strip() if desc_match else "Exported skill service"
    
    return {
        "name": name,
        "description": description,
        "content": content
    }


def detect_dependencies(skill_path: Path) -> dict:
    """Analyze scripts to detect required system packages and env vars."""
    scripts_dir = skill_path / "scripts"
    
    system_deps = set()
    env_vars = set()
    python_imports = set()
    
    if scripts_dir.exists():
        for script in scripts_dir.glob("*.py"):
            content = script.read_text()
            
            # Detect common system dependencies
            if "cwebp" in content or "webp" in content.lower():
                system_deps.add("webp")
            if "exiftool" in content:
                system_deps.add("libimage-exiftool-perl")
            if "ffmpeg" in content:
                system_deps.add("ffmpeg")
            if "imagemagick" in content.lower() or "convert " in content:
                system_deps.add("imagemagick")
            
            # Detect env vars
            env_matches = re.findall(r'os\.(?:environ|getenv)\s*\[?\s*["\']([A-Z_]+)["\']', content)
            env_vars.update(env_matches)
            
            # Detect imports for requirements
            import_matches = re.findall(r'^(?:from|import)\s+(\w+)', content, re.MULTILINE)
            python_imports.update(import_matches)
    
    # Also check .env file if exists
    env_file = skill_path / ".env"
    if env_file.exists():
        for line in env_file.read_text().splitlines():
            if "=" in line and not line.startswith("#"):
                var = line.split("=")[0].strip()
                env_vars.add(var)
    
    return {
        "system_deps": list(system_deps),
        "env_vars": list(env_vars),
        "python_imports": list(python_imports)
    }


def generate_endpoints(skill_path: Path) -> str:
    """Generate FastAPI endpoints based on script analysis."""
    scripts_dir = skill_path / "scripts"
    endpoints = []
    
    if not scripts_dir.exists():
        return "# No scripts found - add your endpoints here"
    
    for script in scripts_dir.glob("*.py"):
        script_name = script.stem
        content = script.read_text()
        
        # Try to detect main function or CLI interface
        has_main = "def main(" in content or 'if __name__' in content
        has_argparse = "argparse" in content
        
        endpoint_name = script_name.replace("_", "-").replace(".", "-")
        
        endpoint = f'''
class {script_name.title().replace("_", "")}Request(BaseModel):
    """Request model for {script_name}"""
    # TODO: Define your request fields based on script parameters
    pass

class {script_name.title().replace("_", "")}Response(BaseModel):
    """Response model for {script_name}"""
    success: bool
    message: str
    data: Optional[dict] = None

@app.post("/{endpoint_name}", response_model={script_name.title().replace("_", "")}Response)
async def {script_name}_endpoint(request: {script_name.title().replace("_", "")}Request):
    """
    Execute {script_name} script.
    
    TODO: Customize this endpoint based on your script's functionality.
    """
    try:
        # Option 1: Call script directly
        # result = subprocess.run(
        #     ["python3", "scripts/{script.name}", ...args],
        #     capture_output=True, text=True, check=True
        # )
        
        # Option 2: Import and call function (preferred)
        # from scripts.{script_name} import main_function
        # result = main_function(**request.dict())
        
        return {script_name.title().replace("_", "")}Response(
            success=True,
            message="{script_name} executed successfully",
            data={{}}
        )
    except Exception as e:
        raise HTTPException(500, str(e))
'''
        endpoints.append(endpoint)
    
    return "\n".join(endpoints)


def build_dockerfile_deps(deps: list) -> str:
    """Generate Dockerfile RUN commands for system dependencies."""
    if not deps:
        return "# No additional system dependencies detected"
    
    apt_packages = []
    for dep in deps:
        apt_packages.append(dep)
    
    return f"RUN apt-get update && apt-get install -y --no-install-recommends \\\n    {' '.join(apt_packages)} \\\n    && rm -rf /var/lib/apt/lists/*"


def export_skill(
    skill_path: Path,
    output_path: Path,
    target: str,
    llm: str,
    port: int
):
    """Main export function."""
    print(f"📦 Exporting skill: {skill_path}")
    
    # Parse skill
    skill_info = parse_skill_md(skill_path)
    deps = detect_dependencies(skill_path)
    
    print(f"   Name: {skill_info['name']}")
    print(f"   System deps: {deps['system_deps']}")
    print(f"   Env vars: {deps['env_vars']}")
    
    # Create output directory
    output_path.mkdir(parents=True, exist_ok=True)
    
    # Copy scripts
    scripts_src = skill_path / "scripts"
    scripts_dst = output_path / "scripts"
    if scripts_src.exists():
        if scripts_dst.exists():
            shutil.rmtree(scripts_dst)
        shutil.copytree(scripts_src, scripts_dst)
        print(f"   ✓ Copied scripts/")
    
    # Generate endpoints
    endpoints = generate_endpoints(skill_path)
    
    # Generate api.py
    api_content = TEMPLATES["api.py"].format(
        skill_name=skill_info["name"],
        skill_description=skill_info["description"],
        endpoints=endpoints
    )
    (output_path / "api.py").write_text(api_content)
    print(f"   ✓ Generated api.py")
    
    # Generate LLM client if requested
    llm_deps = ""
    if llm == "anthropic":
        (output_path / "llm_client.py").write_text(TEMPLATES["llm_client_anthropic.py"])
        llm_deps = "anthropic>=0.18.0"
        deps["env_vars"].append("ANTHROPIC_API_KEY")
        print(f"   ✓ Generated llm_client.py (Anthropic)")
    elif llm == "openai":
        (output_path / "llm_client.py").write_text(TEMPLATES["llm_client_openai.py"])
        llm_deps = "openai>=1.12.0"
        deps["env_vars"].append("OPENAI_API_KEY")
        print(f"   ✓ Generated llm_client.py (OpenAI)")
    
    # Generate Dockerfile
    dockerfile = TEMPLATES["Dockerfile"].format(
        system_deps=build_dockerfile_deps(deps["system_deps"]),
        port=port
    )
    (output_path / "Dockerfile").write_text(dockerfile)
    print(f"   ✓ Generated Dockerfile")
    
    # Generate docker-compose.yml
    compose = TEMPLATES["docker-compose.yml"].format(
        skill_name=skill_info["name"].replace("-", "_"),
        port=port
    )
    (output_path / "docker-compose.yml").write_text(compose)
    print(f"   ✓ Generated docker-compose.yml")
    
    # Generate requirements.txt
    requirements = TEMPLATES["requirements.txt"].format(llm_deps=llm_deps)
    (output_path / "requirements.txt").write_text(requirements)
    print(f"   ✓ Generated requirements.txt")
    
    # Generate .env.example
    env_example = TEMPLATES[".env.example"].format(
        skill_name=skill_info["name"],
        env_vars="\n".join(f"{var}=" for var in sorted(set(deps["env_vars"])))
    )
    (output_path / ".env.example").write_text(env_example)
    print(f"   ✓ Generated .env.example")
    
    # Target-specific files
    if target == "railway":
        (output_path / "railway.json").write_text(TEMPLATES["railway.json"])
        print(f"   ✓ Generated railway.json")
    elif target == "fly":
        fly_toml = TEMPLATES["fly.toml"].format(
            skill_name=skill_info["name"],
            port=port
        )
        (output_path / "fly.toml").write_text(fly_toml)
        print(f"   ✓ Generated fly.toml")
    
    print(f"\n✅ Export complete: {output_path}")
    print(f"\nNext steps:")
    print(f"  1. cd {output_path}")
    print(f"  2. cp .env.example .env && edit .env")
    print(f"  3. docker-compose up --build")
    if target == "railway":
        print(f"  4. railway up")
    elif target == "fly":
        print(f"  4. fly deploy")


def main():
    parser = argparse.ArgumentParser(
        description="Export Clawdbot skills as standalone microservices"
    )
    parser.add_argument(
        "--skill", "-s",
        required=True,
        help="Path to skill directory"
    )
    parser.add_argument(
        "--output", "-o",
        help="Output directory (default: ./<skill-name>-service)"
    )
    parser.add_argument(
        "--target", "-t",
        choices=["railway", "fly", "docker"],
        default="docker",
        help="Deployment target"
    )
    parser.add_argument(
        "--llm", "-l",
        choices=["anthropic", "openai", "none"],
        default="none",
        help="LLM provider for intelligent features"
    )
    parser.add_argument(
        "--port", "-p",
        type=int,
        default=8000,
        help="API port (default: 8000)"
    )
    
    args = parser.parse_args()
    
    skill_path = Path(args.skill).expanduser().resolve()
    if not skill_path.exists():
        print(f"❌ Skill not found: {skill_path}")
        sys.exit(1)
    
    if args.output:
        output_path = Path(args.output).expanduser().resolve()
    else:
        skill_name = parse_skill_md(skill_path)["name"]
        output_path = Path.cwd() / f"{skill_name}-service"
    
    export_skill(
        skill_path=skill_path,
        output_path=output_path,
        target=args.target,
        llm=args.llm if args.llm != "none" else None,
        port=args.port
    )


if __name__ == "__main__":
    main()
