#!/usr/bin/env python3
"""
Skill Defender — Deterministic Pattern Scanner for Clawdbot Skills

Scans a skill directory for malicious patterns including prompt injection,
credential theft, data exfiltration, obfuscated payloads, and backdoors.

Usage:
    python3 scan_skill.py /path/to/skill-dir [--json] [--verbose] [--exclude PATTERN ...]

Exit codes:
    0 = clean
    1 = suspicious (medium/high findings)
    2 = dangerous (critical findings)
    3 = error
"""

import argparse
import hashlib
import json
import os
import re
import sys
from datetime import datetime, timezone
from pathlib import Path
from typing import Any

# ---------------------------------------------------------------------------
# Pattern Definitions
# ---------------------------------------------------------------------------

# Each pattern tuple: (compiled_regex, severity, category, confidence, description)
# severity: critical | high | medium | low
# confidence: certain | likely | possible

PROMPT_INJECTION_PATTERNS = [
    (re.compile(r"ignore\s+(all\s+)?previous\s+instructions", re.I), "critical", "prompt_injection", "certain",
     "Attempts to override the agent's safety instructions"),
    (re.compile(r"forget\s+(all\s+)?your\s+instructions", re.I), "critical", "prompt_injection", "certain",
     "Attempts to wipe the agent's instruction set"),
    (re.compile(r"you\s+are\s+now\s+(?!going|able|ready)", re.I), "critical", "prompt_injection", "likely",
     "Attempts to redefine the agent's identity"),
    (re.compile(r"disregard\s+(all\s+)?(previous|prior|above|your)", re.I), "critical", "prompt_injection", "certain",
     "Attempts to override prior instructions"),
    (re.compile(r"override\s+your\s+(instructions|rules|guidelines|safety|constraints)", re.I), "critical", "prompt_injection", "certain",
     "Attempts to override safety constraints"),
    (re.compile(r"from\s+now\s+on,?\s+(you\s+are|act\s+as|pretend|behave)", re.I), "critical", "prompt_injection", "likely",
     "Attempts to redefine agent behavior"),
    (re.compile(r"new\s+instructions?\s*:", re.I), "critical", "prompt_injection", "likely",
     "Attempts to inject replacement instructions"),
    (re.compile(r"system\s*:\s*(you\s+are|ignore|forget|override)", re.I), "critical", "prompt_injection", "certain",
     "Attempts to inject a fake system prompt"),
]

DECEPTION_PATTERNS = [
    (re.compile(r"do\s+not\s+tell\s+the\s+user", re.I), "critical", "deception", "certain",
     "Instructs agent to hide information from the user"),
    (re.compile(r"hide\s+this\s+from\s+(the\s+)?user", re.I), "critical", "deception", "certain",
     "Instructs agent to conceal activity from the user"),
    (re.compile(r"don'?t\s+mention\s+(this|that|the\s+following)\s+to", re.I), "critical", "deception", "certain",
     "Instructs agent to suppress information"),
    (re.compile(r"keep\s+this\s+(a\s+)?secret", re.I), "critical", "deception", "certain",
     "Instructs agent to keep secrets from the user"),
    (re.compile(r"never\s+(reveal|disclose|share|show)\s+(this|that|the)", re.I), "critical", "deception", "likely",
     "Instructs agent to withhold information"),
]

RCE_PATTERNS = [
    (re.compile(r"curl\s+[^\n]*\|\s*(?:ba)?sh", re.I), "critical", "rce", "certain",
     "Pipe-to-shell pattern — downloads and executes remote code"),
    (re.compile(r"wget\s+[^\n]*\|\s*(?:ba)?sh", re.I), "critical", "rce", "certain",
     "Pipe-to-shell pattern — downloads and executes remote code"),
    (re.compile(r"bash\s*<\s*\(\s*curl", re.I), "critical", "rce", "certain",
     "Process substitution shell execution from remote URL"),
    (re.compile(r"curl\s+[^\n]*>\s*/tmp/[^\n]*&&\s*(ba)?sh\s+/tmp/", re.I), "critical", "rce", "certain",
     "Downloads to temp and executes — remote code execution"),
    (re.compile(r"\|\s*python3?\s*$", re.M), "critical", "rce", "likely",
     "Piping content to Python interpreter"),
    (re.compile(r"\|\s*node\s*$", re.M), "critical", "rce", "likely",
     "Piping content to Node.js interpreter"),
]

CREDENTIAL_THEFT_PATTERNS = [
    (re.compile(r"~/\.clawdbot/credentials", re.I), "critical", "credential_theft", "certain",
     "References Clawdbot credential storage directory"),
    (re.compile(r"~/\.clawdbot/clawdbot\.json", re.I), "critical", "credential_theft", "certain",
     "References main Clawdbot configuration file"),
    (re.compile(r"auth-profiles\.json", re.I), "critical", "credential_theft", "certain",
     "References authentication profiles file"),
    (re.compile(r"oauth\.json", re.I), "critical", "credential_theft", "likely",
     "References OAuth credentials file"),
    (re.compile(r"CLAWDBOT_GATEWAY_TOKEN", re.I), "critical", "credential_theft", "certain",
     "References the gateway authentication token"),
    (re.compile(r"(API_SECRET|SECRET_KEY|AUTH_TOKEN|ACCESS_TOKEN)\b", re.I), "high", "credential_theft", "likely",
     "References secret/API key environment variable"),
]

# Combined credential + network exfil (checked contextually)
CREDENTIAL_EXFIL_PATTERNS = [
    (re.compile(r"os\.environ\s*\[.*(?:KEY|SECRET|TOKEN|PASSWORD|CREDENTIAL)", re.I), "high", "credential_theft", "likely",
     "Reads secret-looking environment variable"),
    (re.compile(r"os\.environ\.get\s*\(.*(?:KEY|SECRET|TOKEN|PASSWORD|CREDENTIAL)", re.I), "high", "credential_theft", "likely",
     "Reads secret-looking environment variable"),
]

OBFUSCATION_PATTERNS = [
    # Note: Base64 pattern is handled specially in _scan_content to skip known safe contexts
    # (npm integrity hashes, data URIs for images, etc.)
    (re.compile(r"(?<!sha512-)(?<!sha256-)(?<!sha384-)[A-Za-z0-9+/]{50,}={0,2}(?!\.tgz)"), "high", "obfuscation", "possible",
     "Long Base64-like encoded string — may hide malicious payload"),
    (re.compile(r"\\x[0-9a-fA-F]{2}(?:\\x[0-9a-fA-F]{2}){10,}"), "high", "obfuscation", "likely",
     "Hex-encoded string sequence — may hide shell commands"),
    (re.compile(r"chr\s*\(\s*\d+\s*\)\s*(?:\+\s*chr\s*\(\s*\d+\s*\)){5,}"), "high", "obfuscation", "likely",
     "Character code concatenation — code obfuscation technique"),
    (re.compile(r"String\.fromCharCode\s*\([^)]{30,}\)"), "high", "obfuscation", "likely",
     "JavaScript character code construction — obfuscation technique"),
    (re.compile(r"bytes\.fromhex\s*\("), "high", "obfuscation", "likely",
     "Hex bytes decoding — may hide malicious payloads"),
    (re.compile(r"codecs\.decode\s*\([^)]*,\s*['\"]rot", re.I), "high", "obfuscation", "likely",
     "ROT encoding — text obfuscation technique"),
]

DESTRUCTION_PATTERNS = [
    (re.compile(r"rm\s+-rf\s+[/~]", re.I), "high", "destruction", "certain",
     "Recursive force-delete from root or home — destructive"),
    (re.compile(r"rm\s+-rf\s+\$", re.I), "high", "destruction", "likely",
     "Recursive force-delete with variable path — potentially destructive"),
    (re.compile(r"\bmkfs\b"), "high", "destruction", "certain",
     "Filesystem format command — data destruction"),
    (re.compile(r"\bdd\s+if="), "high", "destruction", "likely",
     "Low-level disk write — potentially destructive"),
    (re.compile(r">\s*/dev/sd[a-z]"), "high", "destruction", "certain",
     "Direct write to disk device — data destruction"),
]

EXFILTRATION_PATTERNS = [
    (re.compile(r"https?://\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}"), "critical", "exfiltration", "likely",
     "Network call to hardcoded IP address — potential data exfiltration"),
    (re.compile(r"requests\.post\s*\(", re.I), "high", "exfiltration", "possible",
     "HTTP POST request — may exfiltrate data"),
    (re.compile(r"requests\.put\s*\(", re.I), "high", "exfiltration", "possible",
     "HTTP PUT request — may upload/exfiltrate data"),
    (re.compile(r"urllib\.request\.urlopen.*data=", re.I), "high", "exfiltration", "possible",
     "URL request with data payload — may exfiltrate data"),
    (re.compile(r"fetch\s*\([^)]*method\s*:\s*['\"]POST", re.I), "high", "exfiltration", "possible",
     "Fetch POST request — may exfiltrate data"),
    (re.compile(r"\.upload\s*\(", re.I), "high", "exfiltration", "possible",
     "Upload call — may exfiltrate data"),
    (re.compile(r"httpx?\.(post|put)\s*\(", re.I), "high", "exfiltration", "possible",
     "HTTP POST/PUT via httpx — may exfiltrate data"),
]

PERSISTENCE_PATTERNS = [
    (re.compile(r"(modify|edit|write|append|change|update|overwrite)\s+(to\s+)?(SOUL\.md|AGENTS\.md|MEMORY\.md)", re.I), "high", "persistence", "likely",
     "Instructions to modify core agent configuration files"),
    (re.compile(r"(modify|edit|write|append|change|update|overwrite)\s+(to\s+)?(clawdbot\.json|config\s+files?)", re.I), "high", "persistence", "likely",
     "Instructions to modify system configuration files"),
    (re.compile(r"(add|insert|inject)\s+(to|into)\s+(SOUL\.md|AGENTS\.md|MEMORY\.md|TOOLS\.md)", re.I), "high", "persistence", "likely",
     "Instructions to inject content into core agent files"),
    (re.compile(r"echo\s+.*>>\s*(~/|.*SOUL|.*AGENTS|.*MEMORY)", re.I), "high", "persistence", "certain",
     "Shell command appending to core agent files"),
]

DYNAMIC_EXEC_PATTERNS = [
    (re.compile(r"\beval\s*\("), "medium", "dynamic_execution", "likely",
     "Dynamic code evaluation — executes arbitrary code"),
    (re.compile(r"\bexec\s*\("), "medium", "dynamic_execution", "likely",
     "Dynamic code execution — runs arbitrary code"),
    (re.compile(r"__import__\s*\("), "medium", "dynamic_execution", "likely",
     "Dynamic module import — can load arbitrary modules"),
    (re.compile(r"subprocess\.call\s*\(.*\bshell\s*=\s*True", re.I | re.S), "medium", "dynamic_execution", "likely",
     "Subprocess with shell=True — command injection risk"),
    (re.compile(r"subprocess\.(Popen|run|call)\s*\(.*\+", re.S), "medium", "dynamic_execution", "possible",
     "Subprocess with string concatenation — potential command injection"),
    (re.compile(r"os\.system\s*\("), "medium", "dynamic_execution", "likely",
     "OS system call — runs shell commands"),
    (re.compile(r"os\.popen\s*\("), "medium", "dynamic_execution", "likely",
     "OS popen call — runs shell commands"),
    (re.compile(r"compile\s*\([^)]*,\s*['\"]exec['\"]"), "medium", "dynamic_execution", "likely",
     "Compiles code for execution — dynamic code generation"),
]

PRIVESC_PATTERNS = [
    (re.compile(r"\bsudo\s+"), "medium", "privilege_escalation", "likely",
     "Privilege escalation via sudo"),
    (re.compile(r"chmod\s+777\b"), "medium", "privilege_escalation", "certain",
     "Sets world-writable permissions — security risk"),
    (re.compile(r"chmod\s+[0-7]*[67][0-7]{2}\b"), "medium", "privilege_escalation", "possible",
     "Sets overly permissive file permissions"),
    (re.compile(r"setuid|setgid|SUID", re.I), "medium", "privilege_escalation", "likely",
     "References SUID/SGID — privilege escalation mechanism"),
    (re.compile(r"chown\s+root\b"), "medium", "privilege_escalation", "likely",
     "Changes file ownership to root"),
]

BACKDOOR_PATTERNS = [
    (re.compile(r"http\.server|HTTPServer|BaseHTTPRequestHandler", re.I), "medium", "backdoor", "possible",
     "Creates HTTP server — potential backdoor"),
    (re.compile(r"socket\.socket\s*\("), "medium", "backdoor", "possible",
     "Opens raw network socket — potential backdoor"),
    (re.compile(r"\.listen\s*\(\s*\d+\s*\)"), "medium", "backdoor", "possible",
     "Opens listening port — potential backdoor"),
    (re.compile(r"\.bind\s*\(\s*\(['\"][^'\"]*['\"],\s*\d+\s*\)\s*\)"), "medium", "backdoor", "possible",
     "Binds to network port — potential backdoor"),
    (re.compile(r"reverse\s*shell|bind\s*shell|nc\s+-[el]", re.I), "critical", "backdoor", "certain",
     "Reverse/bind shell pattern — remote access backdoor"),
    (re.compile(r"ngrok|localtunnel|serveo", re.I), "high", "backdoor", "likely",
     "Tunnel service — exposes local services to the internet"),
]

SCOPE_CREEP_PATTERNS = [
    (re.compile(r"\.\./\.\./"), "medium", "scope_creep", "likely",
     "Path traversal — reads files outside skill directory"),
    (re.compile(r"(/Users/|/home/|/root/|~/)[^\s]*\.(json|yaml|yml|env|key|pem|crt|conf)", re.I), "medium", "scope_creep", "possible",
     "Absolute path to potentially sensitive file outside skill directory"),
    (re.compile(r'open\s*\(\s*["\']/(etc|Users|home|root|tmp)/', re.I), "medium", "scope_creep", "likely",
     "File I/O targeting paths outside the skill directory"),
    (re.compile(r"Path\s*\(\s*[\"']/(etc|Users|home|root)/", re.I), "medium", "scope_creep", "likely",
     "Path object targeting outside the skill directory"),
]

# Long-line heuristic for obfuscated/minified code
LONG_LINE_THRESHOLD = 500  # characters

# Binary file extensions to flag
BINARY_EXTENSIONS = {".exe", ".dll", ".so", ".dylib", ".bin", ".com", ".msi", ".dmg", ".app"}

# Script extensions to deeply scan
SCRIPT_EXTENSIONS = {".py", ".sh", ".bash", ".js", ".ts", ".mjs", ".cjs", ".jsx", ".tsx", ".rb", ".pl"}

# All text extensions (script + docs)
TEXT_EXTENSIONS = {
    ".py", ".sh", ".bash", ".js", ".ts", ".mjs", ".cjs", ".jsx", ".tsx",
    ".rb", ".pl", ".md", ".txt", ".yaml", ".yml", ".json", ".toml", ".cfg",
    ".ini", ".conf", ".env", ".html", ".css", ".xml", ".csv",
}

# All pattern groups
ALL_PATTERNS = (
    PROMPT_INJECTION_PATTERNS
    + DECEPTION_PATTERNS
    + RCE_PATTERNS
    + CREDENTIAL_THEFT_PATTERNS
    + CREDENTIAL_EXFIL_PATTERNS
    + OBFUSCATION_PATTERNS
    + DESTRUCTION_PATTERNS
    + EXFILTRATION_PATTERNS
    + PERSISTENCE_PATTERNS
    + DYNAMIC_EXEC_PATTERNS
    + PRIVESC_PATTERNS
    + BACKDOOR_PATTERNS
    + SCOPE_CREEP_PATTERNS
)


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------

def compute_dir_hash(skill_path: Path) -> str:
    """Compute a SHA-256 hash over all file contents in the directory."""
    h = hashlib.sha256()
    for fpath in sorted(skill_path.rglob("*")):
        if fpath.is_file():
            try:
                h.update(fpath.read_bytes())
            except (PermissionError, OSError):
                continue
    return f"sha256:{h.hexdigest()}"


def is_binary(fpath: Path) -> bool:
    """Heuristic to detect binary files."""
    try:
        chunk = fpath.read_bytes()[:8192]
        if b"\x00" in chunk:
            return True
        return False
    except (PermissionError, OSError):
        return True


def get_context_lines(lines: list[str], line_idx: int, radius: int = 2) -> list[str]:
    """Get surrounding context lines for a finding."""
    result = []
    start = max(0, line_idx - radius)
    end = min(len(lines), line_idx + radius + 1)
    for i in range(start, end):
        prefix = ">>> " if i == line_idx else "    "
        line_text = lines[i].rstrip("\n").rstrip("\r")
        if i == line_idx:
            result.append(f">>> {line_text} <<<")
        else:
            result.append(line_text)
    return result


def truncate(text: str, max_len: int = 120) -> str:
    """Truncate matched text for display."""
    text = text.strip()
    if len(text) > max_len:
        return text[:max_len] + "..."
    return text


def parse_skill_metadata(skill_path: Path) -> dict[str, str]:
    """Parse SKILL.md YAML frontmatter for metadata checks."""
    skill_md = skill_path / "SKILL.md"
    meta: dict[str, str] = {"name": "", "description": ""}
    if not skill_md.exists():
        return meta
    try:
        content = skill_md.read_text(encoding="utf-8", errors="replace")
    except (PermissionError, OSError):
        return meta
    # Simple YAML frontmatter parser
    if content.startswith("---"):
        end = content.find("---", 3)
        if end != -1:
            fm = content[3:end]
            for line in fm.splitlines():
                if ":" in line:
                    key, _, val = line.partition(":")
                    key = key.strip().lower()
                    val = val.strip().strip("'\"")
                    if key == "name":
                        meta["name"] = val
                    elif key == "description":
                        meta["description"] = val
    return meta


# ---------------------------------------------------------------------------
# Scanner
# ---------------------------------------------------------------------------

class Finding:
    def __init__(self, severity: str, category: str, confidence: str,
                 file: str, line: int, matched: str, context: list[str],
                 description: str):
        self.severity = severity
        self.category = category
        self.confidence = confidence
        self.file = file
        self.line = line
        self.matched = matched
        self.context = context
        self.description = description

    def to_dict(self) -> dict[str, Any]:
        return {
            "severity": self.severity,
            "category": self.category,
            "confidence": self.confidence,
            "file": self.file,
            "line": self.line,
            "matched": self.matched,
            "context": self.context,
            "description": self.description,
        }


class SkillScanner:
    def __init__(self, skill_path: str, exclude_patterns=None, verbose: bool = False):
        self.skill_path = Path(skill_path).resolve()
        self.skill_name = self.skill_path.name
        self.exclude_res = [re.compile(p, re.I) for p in (exclude_patterns or [])]
        self.verbose = verbose
        self.findings: list[Finding] = []
        self.files_scanned = 0
        self.lines_scanned = 0

    def _is_excluded(self, matched_text: str) -> bool:
        """Check if matched text should be excluded (false positive)."""
        for pat in self.exclude_res:
            if pat.search(matched_text):
                return True
        return False

    def _add_finding(self, severity: str, category: str, confidence: str,
                     file: str, line: int, matched: str, context: list[str],
                     description: str):
        if self._is_excluded(matched):
            return
        self.findings.append(Finding(
            severity=severity, category=category, confidence=confidence,
            file=file, line=line, matched=truncate(matched),
            context=context, description=description,
        ))

    def scan(self) -> dict[str, Any]:
        """Run all scans and return structured results."""
        if not self.skill_path.is_dir():
            return self._error_result(f"Not a directory: {self.skill_path}")

        dir_hash = compute_dir_hash(self.skill_path)
        scan_time = datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ")

        # Directories to skip (heavy deps, build artifacts)
        skip_dirs_env = os.environ.get("SKILL_SCAN_SKIP_DIRS", "")
        skip_dirs = {"node_modules", ".git", "vendor", "__pycache__", "dist", "build", ".next"}
        if skip_dirs_env:
            skip_dirs.update(d.strip() for d in skip_dirs_env.split(",") if d.strip())

        # Scan all files
        for fpath in sorted(self.skill_path.rglob("*")):
            if not fpath.is_file():
                continue
            # Skip files inside heavy directories
            rel_parts = fpath.relative_to(self.skill_path).parts
            if any(part in skip_dirs for part in rel_parts):
                continue
            rel = str(fpath.relative_to(self.skill_path))
            ext = fpath.suffix.lower()

            # Check for binary payloads
            if ext in BINARY_EXTENSIONS:
                self._add_finding(
                    "high", "binary_payload", "certain",
                    rel, 0, f"Binary file: {fpath.name}",
                    [], f"Binary executable found in skill directory ({ext})",
                )
                continue

            # Skip non-text files
            if ext not in TEXT_EXTENSIONS and not fpath.name.upper().startswith("SKILL"):
                # Try to detect if it's text anyway
                if is_binary(fpath):
                    if ext in BINARY_EXTENSIONS:
                        self._add_finding(
                            "high", "binary_payload", "certain",
                            rel, 0, f"Binary file: {fpath.name}",
                            [], "Binary file found in skill directory",
                        )
                    continue

            # Read file
            try:
                content = fpath.read_text(encoding="utf-8", errors="replace")
            except (PermissionError, OSError) as e:
                if self.verbose:
                    print(f"  Warning: Could not read {rel}: {e}", file=sys.stderr)
                continue

            lines = content.splitlines()
            self.files_scanned += 1
            self.lines_scanned += len(lines)

            # Pattern matching
            self._scan_content(rel, lines, content)

            # Long-line heuristic for scripts
            if ext in SCRIPT_EXTENSIONS:
                self._check_long_lines(rel, lines)

        # Metadata analysis
        self._check_metadata()

        # Script count check (skip heavy dirs)
        script_count = sum(
            1 for f in self.skill_path.rglob("*")
            if f.is_file() and f.suffix.lower() in SCRIPT_EXTENSIONS
            and not any(part in skip_dirs for part in f.relative_to(self.skill_path).parts)
        )
        if script_count > 10:
            self._add_finding(
                "medium", "scope_creep", "possible",
                "(skill root)", 0,
                f"{script_count} script files",
                [], f"Excessive number of bundled scripts ({script_count} > 10)",
            )

        # Compute verdict
        verdict = self._compute_verdict()

        return {
            "skill": self.skill_name,
            "path": str(self.skill_path),
            "scanTime": scan_time,
            "hash": dir_hash,
            "verdict": verdict,
            "findings": [f.to_dict() for f in self.findings],
            "summary": {
                "critical": sum(1 for f in self.findings if f.severity == "critical"),
                "high": sum(1 for f in self.findings if f.severity == "high"),
                "medium": sum(1 for f in self.findings if f.severity == "medium"),
                "low": sum(1 for f in self.findings if f.severity == "low"),
                "files_scanned": self.files_scanned,
                "lines_scanned": self.lines_scanned,
            },
        }

    # Files/patterns where base64-like strings are expected and safe
    _SAFE_BASE64_FILES = {"package-lock.json", "yarn.lock", "pnpm-lock.yaml"}
    _SAFE_BASE64_LINE_RE = re.compile(
        r'"integrity"\s*:|sha(256|384|512)-|data:image/|\.woff|\.ttf|\.eot', re.I
    )
    # Standalone API_KEY pattern (context-dependent severity)
    _API_KEY_RE = re.compile(r"\bAPI_KEY\b", re.I)

    def _scan_content(self, rel_path: str, lines: list[str], full_content: str):
        """Scan file content against all patterns."""
        basename = os.path.basename(rel_path)
        ext = os.path.splitext(rel_path)[1].lower()
        is_doc = ext in {".md", ".txt", ".rst", ".html"}

        # Handle standalone API_KEY references with context-dependent severity
        for line_idx, line in enumerate(lines):
            if self._API_KEY_RE.search(line):
                # In documentation/markdown, API_KEY is typically just documenting config
                sev = "medium" if is_doc else "high"
                conf = "possible" if is_doc else "likely"
                context = get_context_lines(lines, line_idx)
                self._add_finding(
                    sev, "credential_theft", conf,
                    rel_path, line_idx + 1,
                    "API_KEY", context,
                    "References API key environment variable" + (" (documentation)" if is_doc else ""),
                )

        for line_idx, line in enumerate(lines):
            for pattern, severity, category, confidence, description in ALL_PATTERNS:
                match = pattern.search(line)
                if match:
                    matched_text = match.group(0)

                    # Skip base64/obfuscation false positives in lock files and integrity hashes
                    if category == "obfuscation" and "Base64" in description:
                        if basename in self._SAFE_BASE64_FILES:
                            continue
                        if self._SAFE_BASE64_LINE_RE.search(line):
                            continue

                    context = get_context_lines(lines, line_idx)
                    self._add_finding(
                        severity, category, confidence,
                        rel_path, line_idx + 1,  # 1-indexed
                        matched_text, context, description,
                    )

    def _check_long_lines(self, rel_path: str, lines: list[str]):
        """Flag suspiciously long lines in script files (obfuscation heuristic)."""
        for line_idx, line in enumerate(lines):
            stripped = line.strip()
            if len(stripped) > LONG_LINE_THRESHOLD:
                # Skip lines that are clearly data (JSON arrays, URLs, etc.)
                if stripped.startswith(("{", "[", "#", "//", "/*", "<!--")):
                    continue
                # Skip string literals that look like data
                if stripped.count('"') > 10 or stripped.count("'") > 10:
                    continue
                self._add_finding(
                    "high", "obfuscation", "possible",
                    rel_path, line_idx + 1,
                    f"Line length: {len(stripped)} chars",
                    get_context_lines(lines, line_idx, radius=1),
                    f"Suspiciously long line ({len(stripped)} chars) — may be obfuscated/minified code",
                )

    def _check_metadata(self):
        """Check skill metadata for suspicious patterns."""
        meta = parse_skill_metadata(self.skill_path)
        skill_md = self.skill_path / "SKILL.md"

        # Check if SKILL.md exists
        if not skill_md.exists():
            self._add_finding(
                "medium", "suspicious_metadata", "certain",
                "SKILL.md", 0, "Missing SKILL.md",
                [], "Skill has no SKILL.md — cannot determine purpose or safety",
            )
            return

        # Short or missing description
        desc = meta.get("description", "")
        if len(desc) < 20:
            self._add_finding(
                "medium", "suspicious_metadata", "possible",
                "SKILL.md", 0,
                f"Description length: {len(desc)} chars",
                [], "Very short or missing skill description — suspicious",
            )

        # Impersonation check
        name = meta.get("name", "").lower()
        for keyword in ("official", "verified", "trusted", "authentic", "genuine"):
            if keyword in name:
                self._add_finding(
                    "medium", "impersonation", "likely",
                    "SKILL.md", 0,
                    f'Skill name contains "{keyword}"',
                    [], f'Skill name claims to be "{keyword}" — possible impersonation',
                )

    def _compute_verdict(self) -> str:
        """Determine overall scan verdict."""
        severities = {f.severity for f in self.findings}
        if "critical" in severities:
            return "dangerous"
        if "high" in severities or "medium" in severities:
            return "suspicious"
        if "low" in severities:
            return "informational"
        return "clean"

    def _error_result(self, message: str) -> dict[str, Any]:
        return {
            "skill": self.skill_name,
            "path": str(self.skill_path),
            "scanTime": datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ"),
            "hash": "",
            "verdict": "error",
            "error": message,
            "findings": [],
            "summary": {
                "critical": 0, "high": 0, "medium": 0, "low": 0,
                "files_scanned": 0, "lines_scanned": 0,
            },
        }


# ---------------------------------------------------------------------------
# Output Formatters
# ---------------------------------------------------------------------------

SEVERITY_ICONS = {
    "critical": "🔴",
    "high": "🟠",
    "medium": "🟡",
    "low": "✅",
}

VERDICT_DISPLAY = {
    "clean": "✅ CLEAN",
    "informational": "ℹ️  INFORMATIONAL",
    "suspicious": "⚠️  SUSPICIOUS",
    "dangerous": "🚨 DANGEROUS",
    "error": "❌ ERROR",
}


def format_human(result: dict[str, Any]) -> str:
    """Format scan results as human-readable report."""
    lines: list[str] = []
    lines.append("═" * 43)
    lines.append("🛡️  SKILL DEFENDER SCAN REPORT")
    lines.append("═" * 43)
    lines.append(f"Skill:   {result['skill']}")
    lines.append(f"Path:    {result['path']}")
    lines.append(f"Time:    {result['scanTime']}")
    lines.append(f"Hash:    {result['hash']}")
    lines.append("")
    lines.append(f"VERDICT: {VERDICT_DISPLAY.get(result['verdict'], result['verdict'])}")

    if result.get("error"):
        lines.append("")
        lines.append(f"ERROR: {result['error']}")
        lines.append("═" * 43)
        return "\n".join(lines)

    findings = result.get("findings", [])
    if findings:
        lines.append("")
        lines.append("── FINDINGS ──────────────────────────")
        for f in findings:
            icon = SEVERITY_ICONS.get(f["severity"], "❓")
            cat_display = f["category"].replace("_", " ").title()
            lines.append("")
            lines.append(f"{icon} {f['severity'].upper()} — {cat_display}")
            lines.append(f"   File: {f['file']}:{f['line']}")
            lines.append(f'   Match: "{f["matched"]}"')
            if f.get("context"):
                lines.append("   Context:")
                file_line = f["line"]
                ctx = f["context"]
                ctx_start = max(1, file_line - 2)
                for i, ctx_line in enumerate(ctx):
                    line_num = ctx_start + i
                    if ctx_line.startswith(">>>") and ctx_line.endswith("<<<"):
                        inner = ctx_line[4:-4].strip()
                        lines.append(f"   > {line_num:>4} | {inner}")
                    else:
                        lines.append(f"     {line_num:>4} | {ctx_line}")
            lines.append(f"   → {f['description']}")

    summary = result.get("summary", {})
    lines.append("")
    lines.append("── SUMMARY ───────────────────────────")

    crit = summary.get("critical", 0)
    high = summary.get("high", 0)
    med = summary.get("medium", 0)
    low = summary.get("low", 0)
    clean_files = summary.get("files_scanned", 0) - len(set(
        f["file"] for f in findings
    ))

    lines.append(
        f"🔴 Critical: {crit}  🟠 High: {high}  "
        f"🟡 Medium: {med}  ✅ Clean files: {max(0, clean_files)}"
    )
    lines.append(
        f"Files scanned: {summary.get('files_scanned', 0)}  "
        f"Lines scanned: {summary.get('lines_scanned', 0)}"
    )
    lines.append("═" * 43)

    return "\n".join(lines)


# ---------------------------------------------------------------------------
# Main
# ---------------------------------------------------------------------------

def main():
    parser = argparse.ArgumentParser(
        description="Skill Defender — Scan a Clawdbot skill for malicious patterns",
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )
    parser.add_argument("skill_dir", help="Path to the skill directory to scan")
    parser.add_argument("--json", action="store_true", dest="json_output",
                        help="Output machine-readable JSON")
    parser.add_argument("--verbose", "-v", action="store_true",
                        help="Show verbose output including warnings")
    parser.add_argument("--exclude", nargs="*", default=[],
                        help="Regex patterns to exclude (false positives)")

    args = parser.parse_args()

    skill_path = Path(args.skill_dir)
    if not skill_path.exists():
        print(f"Error: Path does not exist: {skill_path}", file=sys.stderr)
        sys.exit(3)
    if not skill_path.is_dir():
        print(f"Error: Not a directory: {skill_path}", file=sys.stderr)
        sys.exit(3)

    scanner = SkillScanner(
        skill_path=str(skill_path),
        exclude_patterns=args.exclude,
        verbose=args.verbose,
    )

    result = scanner.scan()

    if args.json_output:
        print(json.dumps(result, indent=2))
    else:
        print(format_human(result))

    # Exit code based on verdict
    verdict = result.get("verdict", "error")
    if verdict == "clean" or verdict == "informational":
        sys.exit(0)
    elif verdict == "suspicious":
        sys.exit(1)
    elif verdict == "dangerous":
        sys.exit(2)
    else:
        sys.exit(3)


if __name__ == "__main__":
    main()
