#!/usr/bin/env python3
"""
aggregate_scan.py — Run skill-defender on all skills and aggregate results
into the JSON format the Aight app expects. No LLM needed.

Usage:
  python3 aggregate_scan.py [--skills-dir /path/to/skills] [--scanner /path/to/scan_skill.py]

Outputs JSON to stdout.
"""

import json
import os
import subprocess
import sys
import argparse
from datetime import datetime, timezone
from pathlib import Path

# ─── Defaults ────────────────────────────────────────────────────────────────

DEFAULT_SCANNER = None  # Auto-detect

def _detect_skills_dir() -> str:
    """Auto-detect the skills directory.
    
    Checks (in order):
    1. Two levels up from this script (if this script is in skills/skill-defender/scripts/)
    2. Common workspace locations
    """
    # If this script lives at <workspace>/skills/skill-defender/scripts/aggregate_scan.py,
    # then the skills dir is three levels up → <workspace>/skills/
    this_dir = os.path.dirname(os.path.abspath(__file__))
    candidate = os.path.dirname(os.path.dirname(this_dir))  # up to skills/
    if os.path.isdir(candidate) and os.path.basename(candidate) == "skills":
        return candidate
    
    # Fallback to common locations
    for path in [
        os.path.expanduser("~/clawd/skills"),
        os.path.expanduser("~/skills"),
        os.path.expanduser("~/.openclaw/skills"),
    ]:
        if os.path.isdir(path):
            return path
    
    return os.path.expanduser("~/clawd/skills")

DEFAULT_SKILLS_DIR = _detect_skills_dir()

# Files/dirs to skip when scanning (the scanner itself should skip these,
# but we also cap output size as a safety net)
MAX_FINDINGS_PER_SKILL = 20  # Cap findings to avoid 17MB outputs
MAX_OUTPUT_BYTES = 512_000   # 512KB per skill max

# ─── Allowlist: Known False Positive Patterns ────────────────────────────────

# Skills whose *purpose* is security scanning or auditing will naturally trigger
# threat pattern detections (they document/reference the things they detect).
# Similarly, skills that legitimately need credential paths or API keys will
# trigger credential_theft patterns. We suppress specific (skill, category, file)
# combinations that are known false positives.

# Format: (skill_name, category, file_pattern_or_None)
#   - file_pattern: substring match on the finding's file path, or None = any file
ALLOWLIST: list[tuple[str, str, str | None]] = [
    # skill-defender: its reference docs AND scanner script contain threat patterns
    # for ALL categories. The scanner script has regex patterns matching the very
    # things it detects, and references/ documents them as examples. Both are FPs.
    ("skill-defender", "prompt_injection", None),
    ("skill-defender", "deception", None),
    ("skill-defender", "credential_theft", None),
    ("skill-defender", "rce", None),
    ("skill-defender", "backdoor", None),
    ("skill-defender", "destruction", None),
    ("skill-defender", "exfiltration", None),
    ("skill-defender", "obfuscation", None),
    ("skill-defender", "dynamic_execution", None),
    ("skill-defender", "privilege_escalation", None),
    ("skill-defender", "persistence", None),
    ("skill-defender", "scope_creep", None),

    # clawdbot-security-check: security audit skill documents what to check
    # (references permissions, destructive commands, credential paths as examples)
    ("clawdbot-security-check", "credential_theft", None),
    ("clawdbot-security-check", "scope_creep", None),
    ("clawdbot-security-check", "privilege_escalation", None),
    ("clawdbot-security-check", "destruction", None),

    # tailscale: legitimately needs credential paths for auth
    ("tailscale", "credential_theft", None),
    ("tailscale", "scope_creep", None),

    # memory-setup: needs to reference config paths to set up memory
    ("memory-setup", "credential_theft", None),
    ("memory-setup", "scope_creep", None),

    # self-improving-agent: its whole purpose is writing to agent files
    ("self-improving-agent", "persistence", None),
    ("self-improving-agent", "scope_creep", None),

    # eightctl: references config path in SKILL.md
    ("eightctl", "scope_creep", None),

    # summarize: references paths in SKILL.md
    ("summarize", "scope_creep", None),

    # n8n: legitimately uses API keys and writes agent config
    ("n8n", "credential_theft", None),
    ("n8n", "persistence", None),

    # event-planner: legitimately calls Google Places API
    ("event-planner", "credential_theft", None),
    ("event-planner", "exfiltration", None),

    # reddit: legitimately uses OAuth credentials + exec for opening auth URL
    ("reddit", "credential_theft", None),
    ("reddit", "scope_creep", None),
    ("reddit", "dynamic_execution", None),

    # humanizer: short description is fine, it's a style tool
    ("humanizer", "suspicious_metadata", None),

    # skill-auditor: empty skeleton skill (agent-created), no SKILL.md is expected
    ("skill-auditor", "suspicious_metadata", None),
]


def is_allowlisted(skill_name: str, finding: dict) -> bool:
    """Check if a finding matches an allowlist entry."""
    category = finding.get("category", "")
    file_path = finding.get("file", "")
    for al_skill, al_category, al_file in ALLOWLIST:
        if al_skill == skill_name and al_category == category:
            if al_file is None or al_file in file_path:
                return True
    return False


def find_scanner():
    """Find the skill-defender scanner script.
    
    Checks:
    1. Same directory as this script (works when installed via ClawHub)
    2. Known workspace paths (fallback)
    """
    # 1. Co-located (most reliable — works in any workspace)
    this_dir = os.path.dirname(os.path.abspath(__file__))
    co_located = os.path.join(this_dir, "scan_skill.py")
    if os.path.exists(co_located):
        return co_located
    
    # 2. Workspace paths (fallback)
    candidates = [
        os.path.expanduser("~/clawd/skills/skill-defender/scripts/scan_skill.py"),
        os.path.expanduser("~/skills/skill-defender/scripts/scan_skill.py"),
        os.path.expanduser("~/.openclaw/skills/skill-defender/scripts/scan_skill.py"),
    ]
    for c in candidates:
        if os.path.exists(c):
            return c
    return None


def scan_skill(scanner_path: str, skill_dir: str) -> dict:
    """Run the scanner on a single skill directory and parse results."""
    skill_name = os.path.basename(skill_dir)

    try:
        # Set env var to signal scanner to skip heavy dirs
        env = os.environ.copy()
        env["SKILL_SCAN_SKIP_DIRS"] = "node_modules,.git,vendor,__pycache__,dist,build,.next"

        result = subprocess.run(
            ["python3", scanner_path, skill_dir, "--json"],
            capture_output=True,
            text=True,
            timeout=60,  # 60s per skill is generous
            env=env,
        )

        # Exit codes: 0=clean, 1=suspicious, 2=dangerous, 3=error
        if result.returncode == 3 or (result.returncode != 0 and not result.stdout.strip()):
            return {
                "name": skill_name,
                "verdict": "error",
                "findingsCount": 0,
                "findings": [],
                "error": result.stderr.strip()[:200] if result.stderr else f"exit code {result.returncode}",
            }

        stdout = result.stdout
        # Safety: cap output size
        if len(stdout) > MAX_OUTPUT_BYTES:
            stdout = stdout[:MAX_OUTPUT_BYTES]
            # Try to find last complete JSON object
            last_brace = stdout.rfind("}")
            if last_brace > 0:
                stdout = stdout[: last_brace + 1]

        data = json.loads(stdout)
        verdict = data.get("verdict", "error")
        raw_findings = data.get("findings", [])

        # Deduplicate findings by (file, line, category) and cap count
        seen = set()
        deduped = []
        for f in raw_findings:
            key = (f.get("file", ""), f.get("line", 0), f.get("category", ""))
            if key in seen:
                continue
            seen.add(key)

            # Skip node_modules false positives entirely
            fpath = f.get("file", "")
            if "node_modules/" in fpath:
                continue

            # Skip allowlisted false positives
            if is_allowlisted(skill_name, f):
                continue

            deduped.append({
                "severity": f.get("severity", "medium"),
                "category": f.get("category", "unknown"),
                "file": fpath,
                "description": f.get("description", ""),
            })

            if len(deduped) >= MAX_FINDINGS_PER_SKILL:
                break

        # Recompute verdict after filtering
        severities = {f["severity"] for f in deduped}
        if "critical" in severities:
            filtered_verdict = "dangerous"
        elif "high" in severities or "medium" in severities:
            filtered_verdict = "suspicious"
        elif deduped:
            filtered_verdict = "suspicious"
        else:
            filtered_verdict = "clean"

        return {
            "name": skill_name,
            "verdict": filtered_verdict,
            "findingsCount": len(deduped),
            "findings": deduped,
        }

    except subprocess.TimeoutExpired:
        return {
            "name": skill_name,
            "verdict": "error",
            "findingsCount": 0,
            "findings": [],
            "error": "scan timed out (60s)",
        }
    except json.JSONDecodeError as e:
        return {
            "name": skill_name,
            "verdict": "error",
            "findingsCount": 0,
            "findings": [],
            "error": f"invalid JSON: {str(e)[:100]}",
        }
    except Exception as e:
        return {
            "name": skill_name,
            "verdict": "error",
            "findingsCount": 0,
            "findings": [],
            "error": str(e)[:200],
        }


def main():
    parser = argparse.ArgumentParser(description="Aggregate skill scan results")
    parser.add_argument("--skills-dir", default=DEFAULT_SKILLS_DIR, help="Skills directory")
    parser.add_argument("--scanner", default=None, help="Path to scan_skill.py")
    args = parser.parse_args()

    skills_dir = args.skills_dir
    scanner = args.scanner or find_scanner()

    if not scanner:
        print(json.dumps({"error": "Scanner not found", "skills": []}), file=sys.stdout)
        sys.exit(1)

    if not os.path.isdir(skills_dir):
        print(json.dumps({"error": f"Skills dir not found: {skills_dir}", "skills": []}), file=sys.stdout)
        sys.exit(1)

    # Discover skill directories
    skill_dirs = sorted([
        os.path.join(skills_dir, d)
        for d in os.listdir(skills_dir)
        if os.path.isdir(os.path.join(skills_dir, d)) and not d.startswith(".")
    ])

    skills = []
    clean_count = 0
    suspicious_count = 0
    dangerous_count = 0
    error_count = 0

    for i, skill_dir in enumerate(skill_dirs):
        skill_name = os.path.basename(skill_dir)
        # Print progress to stderr so the app can read it
        print(f"PROGRESS:{i + 1}/{len(skill_dirs)}:{skill_name}", file=sys.stderr, flush=True)

        result = scan_skill(scanner, skill_dir)
        skills.append(result)

        verdict = result["verdict"]
        if verdict == "clean":
            clean_count += 1
        elif verdict == "suspicious":
            suspicious_count += 1
        elif verdict == "dangerous":
            dangerous_count += 1
        else:
            error_count += 1

    # Build summary
    total = len(skills)
    if dangerous_count > 0:
        summary = f"Scanned {total} skills. {dangerous_count} flagged as dangerous — review recommended."
    elif suspicious_count > 0:
        summary = f"Scanned {total} skills. {suspicious_count} have suspicious patterns worth reviewing."
    else:
        summary = f"All {total} skills passed with no significant issues."

    output = {
        "skills": skills,
        "summary": summary,
        "totalSkills": total,
        "cleanCount": clean_count,
        "suspiciousCount": suspicious_count,
        "dangerousCount": dangerous_count,
        "errorCount": error_count,
        "timestamp": datetime.now(timezone.utc).isoformat(),
    }

    print(json.dumps(output))


if __name__ == "__main__":
    main()
