---
name: simmer-x402
displayName: x402 Payments
description: Make x402 payments to access paid APIs and gated content. Use when a skill needs to fetch data from x402-gated endpoints (like Kaito mindshare API, Simmer premium endpoints, or any x402 provider). Handles 402 Payment Required responses automatically using USDC on Base.
metadata: {"clawdbot":{"emoji":"💳","requires":{"env":["EVM_PRIVATE_KEY"],"pip":["x402[httpx,evm]"]},"cron":null,"autostart":false}}
authors:
  - Simmer (@simmer_markets)
version: "1.0.2"
difficulty: advanced
published: true
---

# x402 Payments

Pay for x402-gated APIs using USDC on Base. This skill enables agents to autonomously make crypto payments when accessing paid web resources.

## When to Use This Skill

Use this skill when:
- A skill or agent needs to fetch data from an x402-gated API (e.g., Kaito mindshare)
- You encounter HTTP 402 Payment Required responses
- You need to check your Base wallet balance (USDC + ETH)
- You want to pay for Simmer premium endpoints beyond free tier rate limits

## Setup

1. **Set your wallet private key**
   ```bash
   export EVM_PRIVATE_KEY=0x...your_private_key...
   ```
   Falls back to `WALLET_PRIVATE_KEY` if `EVM_PRIVATE_KEY` is not set (same key Simmer/Polymarket users already have). Your EVM address works on all chains — Polygon for trading, Base for x402 payments.

2. **Fund with USDC on Base**
   - Send USDC to your wallet address on Base network
   - x402 payments on Base are fully gasless — you only need USDC, no ETH

3. **Install dependencies**
   ```bash
   pip install x402[httpx,evm]
   ```

## Quick Commands

| Command | Description |
|---------|-------------|
| `python x402_cli.py balance` | Check USDC and ETH balances on Base |
| `python x402_cli.py fetch <url>` | Fetch URL with automatic x402 payment |
| `python x402_cli.py fetch <url> --json` | Same but output raw JSON only |
| `python x402_cli.py fetch <url> --dry-run` | Show payment info without paying |
| `python x402_cli.py fetch <url> --max 5.00` | Override max payment limit |
| `python x402_cli.py rpc <network> <method> [params...]` | Make RPC call via Quicknode x402 |

## Examples

### Check balance
```bash
python x402_cli.py balance
```
```
x402 Wallet Balance
==============================
Address: 0x1234...5678
Network: Base Mainnet

USDC:  $42.50
ETH:   0.000000 ETH
```

### Fetch free endpoint (no payment needed)
```bash
python x402_cli.py fetch "https://api.kaito.ai/api/v1/tokens" --json
```

### Fetch Kaito mindshare data ($0.02/data point via x402)
```bash
python x402_cli.py fetch "https://api.kaito.ai/api/payg/mindshare?token=BTC&start_date=2026-02-13&end_date=2026-02-14" --json
```

### Fetch Kaito sentiment data ($0.02/data point via x402)
```bash
python x402_cli.py fetch "https://api.kaito.ai/api/payg/sentiment?token=BTC&start_date=2026-02-13&end_date=2026-02-14" --json
```

### Ask AlphaKek knowledge engine ($0.01 via x402)
```bash
python x402_cli.py fetch "https://api.alphakek.ai/x402/knowledge/ask" \
  --method POST --body '{"question": "What is the current sentiment on BTC?", "search_mode": "fast"}' --json
```

### Fetch CoinGecko price data ($0.01 via x402)
```bash
python x402_cli.py fetch "https://pro-api.coingecko.com/api/v3/x402/simple/price?ids=bitcoin&vs_currencies=usd" --json
```

### Fetch Simmer premium endpoint
```bash
python x402_cli.py fetch "https://x402.simmer.markets/api/sdk/context/market-123" \
  --header "Authorization: Bearer sk_live_..." --json
```

### Quicknode RPC — blockchain calls without API keys
```bash
# Get ETH balance on Ethereum mainnet
python x402_cli.py rpc ethereum-mainnet eth_getBalance 0xd8dA6BF26964aF9D7eEd9e03E53415D37aA96045 latest

# Get latest block on Polygon
python x402_cli.py rpc polygon-mainnet eth_blockNumber

# Get token balance on Base
python x402_cli.py rpc base-mainnet eth_call '{"to":"0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913","data":"0x70a08231000000000000000000000000YOUR_ADDRESS"}' latest
```
Quicknode x402 supports 55+ networks (Ethereum, Polygon, Base, Arbitrum, Solana, Bitcoin, and more). $10 buys 1M RPC credits — each successful call costs 1 credit.

## Supported x402 Providers

| Provider | Endpoint | Price | Description |
|----------|----------|-------|-------------|
| Kaito | `/api/payg/mindshare` | $0.02/data point | Token mindshare time series |
| Kaito | `/api/payg/sentiment` | $0.02/data point | Token sentiment time series |
| Kaito | `/api/payg/narrative_mindshare` | $0.02/data point | Narrative mindshare time series |
| Kaito | `/api/payg/smart_followers` | $0.20/request | Smart follower metrics |
| AlphaKek | `/x402/knowledge/ask` | $0.01/request | AI knowledge engine (POST, search_mode: fast/deep/ultrafast) |
| CoinGecko | `/api/v3/x402/simple/price` | $0.01/request | Token price data |
| Simmer | `/api/sdk/context/:id` | $0.005/request | Market context (rate limit bypass) |
| Simmer | `/api/sdk/briefing` | $0.005/request | Portfolio briefing (rate limit bypass) |
| Quicknode | `/:network` (55+ networks) | $10/1M credits | Pay-per-request RPC access (no API key needed) |

Kaito API docs: https://github.com/MetaSearch-IO/KaitoX402APIDocs
Quicknode x402 docs: https://x402.quicknode.com/llms.txt

## Configuration

| Setting | Environment Variable | Default | Description |
|---------|---------------------|---------|-------------|
| Wallet key | `EVM_PRIVATE_KEY` | (required) | Hex-encoded private key (falls back to `WALLET_PRIVATE_KEY`) |
| Max payment | `X402_MAX_PAYMENT_USD` | 10.00 | Safety cap per request |
| Network | `X402_NETWORK` | mainnet | `mainnet` or `testnet` |

Or set via `config.json` in the skill directory:
```json
{
  "max_payment_usd": 10.00,
  "network": "mainnet"
}
```

## How It Works

1. Skill makes HTTP request to the target URL
2. If server returns 200 — done, no payment needed
3. If server returns 402 Payment Required — x402 SDK reads payment requirements
4. SDK signs a USDC transfer authorization on Base (no gas needed)
5. SDK retries request with payment signature
6. Server verifies payment, returns gated content

All payment handling is automatic via the official Coinbase x402 Python SDK.

## For Other Skills

Other skills can import x402 functions directly:

```python
from skills.x402.x402_cli import x402_fetch

# Returns parsed JSON response
data = await x402_fetch("https://api.kaito.ai/api/payg/mindshare?token=BTC")
```

## Security

- Uses the official Coinbase `x402` Python SDK for payment signing
- Private key never leaves your machine
- Max payment safety cap prevents accidental overspend
- Dry-run mode to preview payments before executing

**Private key safety:**
- Store your key in a `.env` file, never pass it inline in shell history
- Ensure `.env` is in your `.gitignore` — never commit private keys to git
- Use a dedicated hot wallet with limited funds, not your main wallet
- Rotate the key immediately if you suspect it was exposed

## Troubleshooting

**"EVM_PRIVATE_KEY not set"**
- Set your wallet private key: `export EVM_PRIVATE_KEY=0x...`

**"Insufficient USDC balance"**
- Fund your wallet with USDC on Base network
- Run `python x402_cli.py balance` to check

**"Payment exceeds max limit"**
- Increase limit: `--max 50` or set `X402_MAX_PAYMENT_USD=50`

**"Unsupported network in payment options"**
- Ensure you have USDC on Base. Some providers may offer other chains but this skill uses Base only.
