"""
SignalHire connector service
============================

This module exposes a small Flask application used by the SignalHire skill to
receive asynchronous callback payloads from the SignalHire Person API and
persist them to CSV files.  It also provides a simple status endpoint
allowing the agent to determine when a given enrichment job is complete.

Usage example:

    export SIGNALHIRE_OUTPUT_DIR=/opt/openclaw/data/signalhire
    python3 -m signalhire.connector.main --port 8787

You should expose the callback endpoint publicly via a reverse proxy or
Cloudflare Tunnel and set `SIGNALHIRE_CALLBACK_URL` accordingly.  For more
information please see the README.md in the parent directory.
"""

from __future__ import annotations

import argparse
import csv
import datetime as _dt
import json
import os
import pathlib
import threading
from typing import Any, Dict, List, Optional

from flask import Flask, jsonify, request

_LOCK = threading.Lock()


def create_app(output_dir: Optional[str] = None) -> Flask:
    """Create and configure the Flask application.

    Args:
        output_dir: Directory where CSV files will be written.  If None,
            the environment variable `SIGNALHIRE_OUTPUT_DIR` is consulted,
            defaulting to ``./data/signalhire``.

    Returns:
        Configured Flask application.
    """
    app = Flask(__name__)
    out_dir = output_dir or os.environ.get("SIGNALHIRE_OUTPUT_DIR") or os.path.join(
        os.getcwd(), "data", "signalhire"
    )
    # Ensure the directory exists
    pathlib.Path(out_dir).mkdir(parents=True, exist_ok=True)

    def _derive_input_type(value: str) -> str:
        """Infer input type from the item string."""
        if not value:
            return "unknown"
        if value.startswith("http://") or value.startswith("https://"):
            return "linkedin"
        if "@" in value:
            return "email"
        # Very simple phone heuristic: contains plus or digits only
        digits = [c for c in value if c.isdigit()]
        if len(digits) > 5:
            return "phone"
        return "uid"

    def _write_rows(request_id: str, rows: List[Dict[str, Any]]) -> None:
        """Write rows to per‑job and consolidated CSV files.

        This function acquires a global lock to prevent interleaved writes from
        concurrent requests.
        """
        per_request = os.path.join(out_dir, f"results_{request_id}.csv")
        consolidated = os.path.join(out_dir, "results_all.csv")
        # Define CSV header order
        fieldnames = [
            "request_id",
            "input_type",
            "input_value",
            "status",
            "full_name",
            "title",
            "company_name",
            "location",
            "linkedin_url",
            "emails",
            "phones",
            "source",
            "received_at_utc",
        ]
        with _LOCK:
            # Write per-request file
            for path in (per_request, consolidated):
                # Create file and write header if necessary
                write_header = not os.path.exists(path)
                with open(path, "a", newline="", encoding="utf-8") as f:
                    writer = csv.DictWriter(f, fieldnames=fieldnames)
                    if write_header:
                        writer.writeheader()
                    for row in rows:
                        writer.writerow(row)

    def _join_values(value: Any) -> str:
        """Join list or scalar values into a semicolon‑separated string."""
        if not value:
            return ""
        if isinstance(value, list):
            return ";".join(str(x) for x in value if x)
        return str(value)

    def _extract_location(candidate: Dict[str, Any]) -> str:
        """Extract a single location name from the candidate data."""
        loc = candidate.get("location") or candidate.get("locations") or []
        if isinstance(loc, list):
            # The documentation shows locations as a list of objects with a name
            names: List[str] = []
            for entry in loc:
                if isinstance(entry, str):
                    names.append(entry)
                elif isinstance(entry, dict) and entry.get("name"):
                    names.append(entry.get("name"))
            return ";".join(names)
        return str(loc) if loc else ""

    @app.route("/signalhire/callback", methods=["POST"])
    def handle_callback() -> Any:
        """Endpoint to receive SignalHire callback payloads.

        The body should be JSON containing either a list of items or a single
        dictionary.  If a top-level ``requestId`` field is present it will be
        used for the CSV file name; otherwise the ``Request-Id`` header is
        inspected.  Each object in the payload must contain at least an
        ``item`` field (renamed to ``input_value``) and a ``status`` field.
        Additional candidate data is extracted if present.
        """
        # Try to parse JSON payload; fallback to raw string if necessary
        try:
            payload = request.get_json(force=True, silent=True)
        except Exception:
            payload = None
        # Determine request ID from body or headers
        request_id: Optional[str] = None
        if isinstance(payload, dict) and "requestId" in payload:
            request_id = str(payload.get("requestId"))
        # The Request-Id header is used by SignalHire for correlation
        if request_id is None:
            request_id = request.headers.get("Request-Id")
        if request_id is None:
            # Fallback to unknown; results_all.csv will still receive rows
            request_id = "unknown"

        # Normalise payload into a list of items
        items: List[Dict[str, Any]]
        if isinstance(payload, list):
            items = payload
        elif isinstance(payload, dict):
            # If the body is a dict, check if it has 'items' or 'results'
            if isinstance(payload.get("items"), list):
                items = payload["items"]
            elif isinstance(payload.get("results"), list):
                items = payload["results"]
            else:
                items = [payload]
        else:
            # Unsupported format
            return jsonify({"error": "Invalid payload format"}), 400

        now_utc = _dt.datetime.utcnow().replace(tzinfo=_dt.timezone.utc).isoformat()
        rows: List[Dict[str, Any]] = []
        for item in items:
            if not isinstance(item, dict):
                continue
            raw_item = item.get("item") or item.get("input") or item.get("id")
            status = item.get("status") or "unknown"
            candidate = item.get("candidate") or {}
            # Derive row
            row = {
                "request_id": request_id,
                "input_type": _derive_input_type(str(raw_item) if raw_item is not None else ""),
                "input_value": raw_item,
                "status": status,
                "full_name": candidate.get("fullName") or candidate.get("name") or "",
                "title": candidate.get("title") or candidate.get("position") or "",
                "company_name": candidate.get("company")
                or candidate.get("companyName")
                or "",
                "location": _extract_location(candidate),
                "linkedin_url": candidate.get("linkedinUrl")
                or candidate.get("linkedin")
                or "",
                "emails": _join_values(candidate.get("emails") or candidate.get("email")),
                "phones": _join_values(candidate.get("phones") or candidate.get("phone")),
                "source": "signalhire",
                "received_at_utc": now_utc,
            }
            rows.append(row)

        # Write rows to disk
        if rows:
            _write_rows(request_id, rows)

        # Return HTTP 200 quickly as required by SignalHire
        return jsonify({"status": "ok", "rows": len(rows)}), 200

    @app.route("/signalhire/jobs/<request_id>", methods=["GET"])
    def job_status(request_id: str) -> Any:
        """Return job status for a given request ID.

        The response includes a boolean ``ready`` flag and the current row count.
        A job is considered ready when at least one row has been written for
        that request ID.  Agents should compare the row count to the number
        of items sent in the original Person API request to determine
        completeness.
        """
        per_request = os.path.join(out_dir, f"results_{request_id}.csv")
        if not os.path.exists(per_request):
            return jsonify({"ready": False, "rows": 0})
        # Count rows (excluding header)
        try:
            with open(per_request, "r", encoding="utf-8") as f:
                row_count = sum(1 for _ in f) - 1
        except Exception:
            row_count = 0
        return jsonify({"ready": row_count > 0, "rows": max(0, row_count)})

    #

    return app


def main() -> None:
    parser = argparse.ArgumentParser(description="SignalHire callback connector")
    parser.add_argument("--host", default="0.0.0.0", help="Host to listen on")
    parser.add_argument("--port", type=int, default=8787, help="Port to listen on")
    parser.add_argument("--output-dir", help="Override output directory")
    args = parser.parse_args()
    app = create_app(args.output_dir)
    app.run(host=args.host, port=args.port)


if __name__ == "__main__":
    main()