#!/usr/bin/env python3
import argparse, json, re, subprocess, sys

def run(cmd):
    p = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
    if p.returncode != 0:
        raise RuntimeError(f"command failed ({p.returncode}): {' '.join(cmd)}\n{p.stderr.strip()}")
    return p.stdout

def norm(s: str) -> str:
    s = (s or "").strip().lower()
    s = re.sub(r"\s+", " ", s)
    return s

def is_e164(s: str) -> bool:
    return bool(re.fullmatch(r"\+[1-9]\d{6,14}", (s or "").strip()))

FIELDS = ["name", "givenName", "familyName", "nickName", "nickGivenName", "nickFamilyName", "username", "number", "uuid"]

def score_contact(c: dict, qn: str) -> int:
    # higher is better
    best = 0
    for f in FIELDS:
        v = c.get(f)
        if not v:
            continue
        vn = norm(str(v))
        if not vn:
            continue
        if vn == qn:
            best = max(best, 100)
        elif qn and vn.startswith(qn):
            best = max(best, 60)
        elif qn and qn in vn:
            best = max(best, 40)
    return best

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--account", required=True, help="Sender account number (E.164), used as -u")
    ap.add_argument("--query", required=True, help="Name/nickname/number fragment to search")
    ap.add_argument("--limit", type=int, default=20)
    args = ap.parse_args()

    q = args.query.strip()
    qn = norm(q)

    out = run(["signal-cli", "-o", "json", "-u", args.account, "listContacts"])
    contacts = json.loads(out)

    matches = []

    if is_e164(q):
        for c in contacts:
            if c.get("number") == q:
                matches.append({"score": 100, "contact": c})
        if not matches:
            # allow sending to unlisted numbers too; still show nothing here
            pass
    else:
        for c in contacts:
            s = score_contact(c, qn)
            if s > 0:
                matches.append({"score": s, "contact": c})

    matches.sort(key=lambda m: (-m["score"], norm(m["contact"].get("name") or ""), m["contact"].get("number") or ""))
    matches = matches[: args.limit]

    # output compact JSON list
    simplified = []
    for m in matches:
        c = m["contact"]
        simplified.append({
            "score": m["score"],
            "number": c.get("number"),
            "name": c.get("name"),
            "nickName": c.get("nickName"),
            "givenName": c.get("givenName"),
            "familyName": c.get("familyName"),
            "uuid": c.get("uuid"),
            "username": c.get("username"),
            "isBlocked": c.get("isBlocked"),
            "unregistered": c.get("unregistered"),
        })

    print(json.dumps(simplified, ensure_ascii=False, indent=2))

if __name__ == "__main__":
    try:
        main()
    except Exception as e:
        print(str(e), file=sys.stderr)
        sys.exit(2)
