---
name: sideload-avatar-generator
description: Generate 3D avatars (VRM/GLB/MML) from text prompts or images via Sideload.gg. Pay-per-use with any x402 wallet (USDC on Base).
metadata: {"openclaw":{"emoji":"🎭","requires":{"bins":["node"]}}}
---

# Sideload Avatar Generator

Generate 3D avatars from text prompts or reference images using [Sideload.gg](https://sideload.gg). Pay-per-use via the [x402 protocol](https://x402.org) — $2 USDC per generation on Base.

**Works with any x402 wallet.** Bring your own wallet and private key — no proprietary wallet required.

## What You Get

Each generation produces four formats:

| Format | File | Use Case |
|--------|------|----------|
| **GLB** | `.glb` | Universal 3D — Three.js, Unity, Unreal, web viewers |
| **VRM** | `.vrm` | Avatar standard — VRChat, VTubing, social apps |
| **MML** | URL | Metaverse Markup Language — for [MML-compatible](https://mml.io) worlds |
| **PNG** | `.png` | Processed reference image used for generation |

## 🎭 Rendering Avatars with @pixiv/three-vrm

The VRM output is designed to work with [@pixiv/three-vrm](https://github.com/pixiv/three-vrm) — the standard Three.js library for loading, displaying, and animating VRM avatars. If you're already building with Three.js, generated avatars plug right in with full skeleton support:

```javascript
import * as THREE from 'three';
import { GLTFLoader } from 'three/examples/jsm/loaders/GLTFLoader';
import { VRMLoaderPlugin } from '@pixiv/three-vrm';

const loader = new GLTFLoader();
loader.register((parser) => new VRMLoaderPlugin(parser));

loader.load('https://aiml.sideload.gg/models/avt-xxx.vrm', (gltf) => {
  const vrm = gltf.userData.vrm;
  scene.add(vrm.scene);

  // Animate bone transforms, look-at, etc.
});
```

This makes it easy to generate an avatar with Sideload and immediately use it in any Three.js scene — games, social apps, virtual worlds, VTubing, and more.

For more on building interactive 3D experiences in the metaverse, see [awesome-mml](https://github.com/DirectiveCreator/awesome-mml) — a curated list of MML (Metaverse Markup Language) resources.

## Prerequisites

- **Node.js 18+**
- **An x402 payment token** — sign a payment with your own wallet/signer and pass it via `--x402-token`. This skill never handles private keys.

  ```bash
  # Check the cost first
  node scripts/generate.js --probe

  # Generate with your x402 token
  node scripts/generate.js --prompt "..." --x402-token <base64-encoded-payment>
  ```

  Use any x402-compatible client to obtain a payment token: [Coinbase x402 SDK](https://github.com/coinbase/x402), [Thirdweb x402](https://portal.thirdweb.com/payments/x402), or your own signing flow.

## Setup

```bash
npm install
```

## Usage

### Generate from Text Prompt

```bash
node scripts/generate.js --prompt "A cyberpunk samurai with glowing red armor" --x402-token <token>
```

### Generate from Image URL

```bash
node scripts/generate.js --image https://example.com/character.png --x402-token <token>
```

### Generate from Local Image

```bash
node scripts/generate.js --image /path/to/photo.jpg --x402-token <token>
```

### Check Cost (No Payment)

```bash
node scripts/generate.js --probe
```

### Check Job Status

```bash
node scripts/status.js avt-a1b2c3d4
```

### Options

| Flag | Description |
|------|-------------|
| `--prompt "text"` | Text description of the avatar |
| `--image <url-or-path>` | Reference image (URL or local file path) |
| `--x402-token <token>` | x402 payment token (required for generation) |
| `--probe` | Check cost without generating |
| `--output <name>` | Custom filename for downloaded files |
| `--no-download` | Skip downloading result files |

## API Reference

See [SIDELOAD-API.md](./SIDELOAD-API.md) for the full API documentation, or visit [sideload.gg/agents/raw](https://sideload.gg/agents/raw).

### Quick Reference

**Generate:**
```
POST https://sideload.gg/api/agent/generate
Headers: Content-Type: application/json, x-payment: <x402_token>
```

**Text:** `{ "type": "text", "prompt": "description" }`
**Image:** `{ "type": "image", "imageUrl": "https://..." }`

**Poll:** `GET https://sideload.gg/api/agent/generate/{jobId}/status` (no auth needed)

## Prompt Tips

Be specific about:
- **Appearance:** clothing, colors, accessories
- **Style:** realistic, anime, cartoon, cyberpunk
- **Features:** armor, weapons, hairstyle, wings

Good prompts:
- `"A steampunk engineer with leather tool belt, copper mechanical arm, weathered pilot hat"`
- `"An anime-style sorceress with long silver hair, glowing purple eyes, ornate golden staff"`
- `"A futuristic soldier in white and blue power armor with glowing energy shield"`

## Image Tips

- PNG, JPG, or WebP
- Front-facing portraits or full-body shots work best
- Clear outlines and distinct clothing/features
- Higher resolution → better results

## Rate Limits & Cost

- **$2 USDC** per generation (x402 on Base, chain ID 8453)
- **10 generations** per 30 minutes per wallet
- Check `Retry-After` header on 429 responses

## Links

- [Sideload.gg](https://sideload.gg)
- [API Documentation](https://sideload.gg/agents/raw)
- [@pixiv/three-vrm](https://github.com/pixiv/three-vrm) — Three.js VRM avatar loader
- [awesome-mml](https://github.com/DirectiveCreator/awesome-mml) — MML resources for the metaverse
- [x402 Protocol](https://x402.org)
- [Coinbase x402 SDK](https://github.com/coinbase/x402)
- [VRM Specification](https://vrm.dev/en/)
- [MML (Metaverse Markup Language)](https://mml.io)
