#!/bin/bash
# Update a checklist task in a Shortcut story
# Usage: ./shortcut-update-task.sh <story-id> <task-id> [--complete|--incomplete]

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

STORY_ID="${1:-}"
TASK_ID="${2:-}"

if [ -z "$STORY_ID" ] || [ -z "$TASK_ID" ]; then
  echo "Usage: $0 <story-id> <task-id> [--complete|--incomplete]"
  exit 1
fi
shift 2

# Parse action
ACTION=""
while [[ $# -gt 0 ]]; do
  case $1 in
    --complete)
      ACTION="complete"
      shift
      ;;
    --incomplete)
      ACTION="incomplete"
      shift
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

if [ -z "$ACTION" ]; then
  echo "Must specify --complete or --incomplete"
  exit 1
fi

# Build payload
if [ "$ACTION" = "complete" ]; then
  PAYLOAD='{"complete": true}'
else
  PAYLOAD='{"complete": false}'
fi

# Update task
RESPONSE=$(curl -s -X PUT \
  -H "Content-Type: application/json" \
  -H "Shortcut-Token: $TOKEN" \
  -d "$PAYLOAD" \
  "$BASE_URL/stories/$STORY_ID/tasks/$TASK_ID")

# Output
DESCRIPTION=$(echo "$RESPONSE" | jq -r '.description')
COMPLETE=$(echo "$RESPONSE" | jq -r '.complete')

if [ "$DESCRIPTION" != "null" ]; then
  STATUS=$([ "$COMPLETE" = "true" ] && echo "✅" || echo "☐")
  echo "$STATUS Updated task #$TASK_ID: $DESCRIPTION"
else
  echo "❌ Failed to update task"
  echo "$RESPONSE" | jq .
  exit 1
fi
