#!/bin/bash
# Update a Shortcut story (move state, mark complete, add description)
# Usage: ./shortcut-update-story.sh <story-id> [options]
#   --complete         Mark as complete
#   --todo             Move to To Do
#   --in-progress      Move to In Progress
#   --description "text"  Update description

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

STORY_ID="${1:-}"
if [ -z "$STORY_ID" ]; then
  echo "Usage: $0 <story-id> [options]"
  exit 1
fi
shift

# Workflow state IDs (can be overridden via env vars or config file)
# To find your workspace's state IDs:
# curl -X GET -H "Shortcut-Token: $SHORTCUT_API_TOKEN" \
#   https://api.app.shortcut.com/api/v3/workflows | jq '.[] | .states[] | "\(.name): \(.id)"'

if [ -f ~/.config/shortcut/workflow-states ]; then
  source ~/.config/shortcut/workflow-states
else
  # Default state IDs (common Shortcut workspace defaults)
  STATE_BACKLOG="${SHORTCUT_STATE_BACKLOG:-500000006}"
  STATE_TODO="${SHORTCUT_STATE_TODO:-500000007}"
  STATE_IN_PROGRESS="${SHORTCUT_STATE_IN_PROGRESS:-500000008}"
  STATE_IN_REVIEW="${SHORTCUT_STATE_IN_REVIEW:-500000009}"
  STATE_DONE="${SHORTCUT_STATE_DONE:-500000010}"
fi

# Build update payload
UPDATES=()

while [[ $# -gt 0 ]]; do
  case $1 in
    --complete)
      UPDATES+=("\"workflow_state_id\": $STATE_DONE")
      shift
      ;;
    --todo)
      UPDATES+=("\"workflow_state_id\": $STATE_TODO")
      shift
      ;;
    --in-progress)
      UPDATES+=("\"workflow_state_id\": $STATE_IN_PROGRESS")
      shift
      ;;
    --description)
      UPDATES+=("\"description\": \"$2\"")
      shift 2
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

if [ ${#UPDATES[@]} -eq 0 ]; then
  echo "No updates specified"
  exit 1
fi

# Join updates with commas
PAYLOAD="{$(IFS=,; echo "${UPDATES[*]}")}"

# Update story
RESPONSE=$(curl -s -X PUT \
  -H "Content-Type: application/json" \
  -H "Shortcut-Token: $TOKEN" \
  -d "$PAYLOAD" \
  "$BASE_URL/stories/$STORY_ID")

# Output
STORY_NAME=$(echo "$RESPONSE" | jq -r '.name')
if [ "$STORY_NAME" != "null" ]; then
  echo "✅ Updated story #$STORY_ID: $STORY_NAME"
else
  echo "❌ Failed to update story"
  echo "$RESPONSE" | jq .
  exit 1
fi
