#!/bin/bash
# Show details of a Shortcut story including checklist items
# Usage: ./shortcut-show-story.sh <story-id>

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

STORY_ID="${1:-}"
if [ -z "$STORY_ID" ]; then
  echo "Usage: $0 <story-id>"
  exit 1
fi

# Fetch story
RESPONSE=$(curl -s -H "Shortcut-Token: $TOKEN" "$BASE_URL/stories/$STORY_ID")

# Check for error
NAME=$(echo "$RESPONSE" | jq -r '.name')
if [ "$NAME" = "null" ]; then
  echo "❌ Story not found or error"
  echo "$RESPONSE" | jq .
  exit 1
fi

# Pretty print
echo "Story #$STORY_ID: $NAME"
echo "Status: $(echo "$RESPONSE" | jq -r 'if .completed then "✅ Completed" else "🔲 Active" end')"
echo "State ID: $(echo "$RESPONSE" | jq -r '.workflow_state_id')"

DESCRIPTION=$(echo "$RESPONSE" | jq -r '.description')
if [ -n "$DESCRIPTION" ] && [ "$DESCRIPTION" != "null" ]; then
  echo ""
  echo "Description:"
  echo "$DESCRIPTION"
fi

# Show checklist tasks
TASK_COUNT=$(echo "$RESPONSE" | jq '.tasks | length')
if [ "$TASK_COUNT" -gt 0 ]; then
  echo ""
  echo "Checklist items:"
  echo "$RESPONSE" | jq -r '.tasks[] | 
    "  [\(.id)] " + (if .complete then "✅" else "☐" end) + " \(.description)"'
fi

# Show comments (exclude deleted)
COMMENT_COUNT=$(echo "$RESPONSE" | jq '[.comments[] | select(.deleted == false)] | length')
if [ "$COMMENT_COUNT" -gt 0 ]; then
  echo ""
  echo "Comments:"
  echo "$RESPONSE" | jq -r '.comments[] | select(.deleted == false) | 
    "  [\(.id)] \(.text)"'
fi
