#!/bin/bash
# List all active Shortcut stories
# Usage: ./shortcut-list-stories.sh [options]
#   --all        Include archived stories
#   --json       Output raw JSON
#   --completed  Show only completed stories
#   --active     Show only active (non-completed) stories

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

# Parse args
SHOW_ARCHIVED=false
OUTPUT_JSON=false
FILTER_COMPLETED=""

while [[ $# -gt 0 ]]; do
  case $1 in
    --all)
      SHOW_ARCHIVED=true
      shift
      ;;
    --json)
      OUTPUT_JSON=true
      shift
      ;;
    --completed)
      FILTER_COMPLETED="true"
      shift
      ;;
    --active)
      FILTER_COMPLETED="false"
      shift
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Build query
if [ "$SHOW_ARCHIVED" = true ]; then
  QUERY='{}'
else
  QUERY='{"archived": false}'
fi

# Fetch stories
RESPONSE=$(curl -s -X POST \
  -H "Content-Type: application/json" \
  -H "Shortcut-Token: $TOKEN" \
  -d "$QUERY" \
  "$BASE_URL/stories/search")

# Output
if [ "$OUTPUT_JSON" = true ]; then
  echo "$RESPONSE" | jq .
  exit 0
fi

# Filter by completion if requested
if [ -n "$FILTER_COMPLETED" ]; then
  RESPONSE=$(echo "$RESPONSE" | jq --arg completed "$FILTER_COMPLETED" '[.[] | select(.completed == ($completed == "true"))]')
fi

# Pretty print
echo "$RESPONSE" | jq -r '.[] | 
  "[\(.id)] \(if .completed then "✅" else "  " end) \(.name)" + 
  (if .workflow_state_id then " (state: \(.workflow_state_id))" else "" end)'

echo ""
echo "Total: $(echo "$RESPONSE" | jq 'length') stories"
