#!/bin/bash
# Initialize workflow state IDs for your Shortcut workspace
# Usage: ./shortcut-init-workflow.sh

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

if [ -z "$TOKEN" ]; then
  echo "❌ SHORTCUT_API_TOKEN not found"
  echo "Set it via environment variable or store in ~/.config/shortcut/api-token"
  exit 1
fi

echo "🔍 Fetching workflow states from your Shortcut workspace..."
echo

STATES=$(curl -s -X GET \
  -H "Content-Type: application/json" \
  -H "Shortcut-Token: $TOKEN" \
  "$BASE_URL/workflows")

if [ $? -ne 0 ]; then
  echo "❌ Failed to fetch workflow states"
  exit 1
fi

# Parse and display states
echo "Available workflow states:"
echo "$STATES" | jq -r '.[] | .states[] | "  \(.name): \(.id)"'
echo

# Generate config file
CONFIG_DIR="$HOME/.config/shortcut"
mkdir -p "$CONFIG_DIR"

cat > "$CONFIG_DIR/workflow-states" << 'EOF'
# Shortcut workflow state IDs for your workspace
# Generated by shortcut-init-workflow.sh
# Edit these to match your workspace's actual state IDs

EOF

echo "$STATES" | jq -r '.[] | .states[] | "export SHORTCUT_STATE_\(.name | ascii_upcase | gsub(" "; "_"))=\(.id)"' >> "$CONFIG_DIR/workflow-states"

echo "✅ Workflow states saved to: $CONFIG_DIR/workflow-states"
echo
echo "To use these states, either:"
echo "  1. Source the file: source $CONFIG_DIR/workflow-states"
echo "  2. Add to ~/.bashrc: echo 'source $CONFIG_DIR/workflow-states' >> ~/.bashrc"
echo "  3. The scripts will auto-load it from $CONFIG_DIR/workflow-states"
echo
echo "Current configuration:"
cat "$CONFIG_DIR/workflow-states"
