#!/bin/bash
# Create a new Shortcut story
# Usage: ./shortcut-create-story.sh "Story name" [--description "text"] [--type feature|bug|chore]

set -euo pipefail

TOKEN="${SHORTCUT_API_TOKEN:-$(cat ~/.config/shortcut/api-token 2>/dev/null | tr -d '\n')}"
BASE_URL="https://api.app.shortcut.com/api/v3"

# Args
STORY_NAME="${1:-}"
DESCRIPTION=""
STORY_TYPE="feature"

if [ -z "$STORY_NAME" ]; then
  echo "Usage: $0 \"Story name\" [--description \"text\"] [--type feature|bug|chore]"
  exit 1
fi
shift

# Parse optional args
while [[ $# -gt 0 ]]; do
  case $1 in
    --description)
      DESCRIPTION="$2"
      shift 2
      ;;
    --type)
      STORY_TYPE="$2"
      shift 2
      ;;
    *)
      echo "Unknown option: $1"
      exit 1
      ;;
  esac
done

# Build JSON payload
PAYLOAD=$(jq -n \
  --arg name "$STORY_NAME" \
  --arg type "$STORY_TYPE" \
  --arg desc "$DESCRIPTION" \
  '{
    name: $name,
    story_type: $type,
    workflow_state_id: 500000006
  } + (if $desc != "" then {description: $desc} else {} end)')

# Create story
RESPONSE=$(curl -s -X POST \
  -H "Content-Type: application/json" \
  -H "Shortcut-Token: $TOKEN" \
  -d "$PAYLOAD" \
  "$BASE_URL/stories")

# Output
STORY_ID=$(echo "$RESPONSE" | jq -r '.id')
STORY_URL=$(echo "$RESPONSE" | jq -r '.app_url')

if [ "$STORY_ID" != "null" ]; then
  echo "✅ Created story #$STORY_ID: $STORY_NAME"
  echo "   $STORY_URL"
else
  echo "❌ Failed to create story"
  echo "$RESPONSE" | jq .
  exit 1
fi
