#!/usr/bin/env node

/**
 * Submit a use case to clawusecase.com
 * 
 * Usage:
 *   node submit.js \
 *     --title "Email notifications for Pro subscriptions" \
 *     --hook "Sends welcome emails automatically" \
 *     --problem "Users weren't getting confirmation emails..." \
 *     --solution "Built Resend integration..." \
 *     --category "Business/SaaS" \
 *     --skills "GitHub,Stripe,Resend" \
 *     --requirements "Resend account, Stripe webhooks" \
 *     --author-username "josephliow" \
 *     --author-handle "josephliow" \
 *     --author-platform "twitter" \
 *     --author-link "https://twitter.com/josephliow"
 * 
 * Or for anonymous:
 *   node submit.js ... --anonymous
 */

const https = require('https');
const fs = require('fs');
const path = require('path');

// Load config
const configPath = path.join(__dirname, 'config.json');
const config = fs.existsSync(configPath) 
  ? JSON.parse(fs.readFileSync(configPath, 'utf8'))
  : {};

// API Configuration
const API_URL = process.env.CLAWUSECASE_API_URL || config.apiUrl || 'clawusecase.com';
const API_PATH = process.env.CLAWUSECASE_API_PATH || config.apiPath || '/api/submissions';

// Parse command line arguments
function parseArgs() {
  const args = process.argv.slice(2);
  const parsed = {};
  
  for (let i = 0; i < args.length; i++) {
    const arg = args[i];
    
    // Handle flags like --anonymous
    if (arg.startsWith('--') && (i + 1 >= args.length || args[i + 1].startsWith('--'))) {
      const key = arg.replace(/^--/, '').replace(/-/g, '_');
      parsed[key] = true;
      continue;
    }
    
    // Handle key-value pairs
    if (arg.startsWith('--')) {
      const key = arg.replace(/^--/, '').replace(/-/g, '_');
      const value = args[i + 1];
      parsed[key] = value;
      i++; // Skip next arg since we consumed it
    }
  }
  
  return parsed;
}

// Validate required fields
function validate(data) {
  const errors = [];
  
  if (!data.title || data.title.length < 20) {
    errors.push('Title must be at least 20 characters');
  }
  if (!data.hook || data.hook.length < 50) {
    errors.push('Hook must be at least 50 characters');
  }
  if (!data.problem || data.problem.length < 100) {
    errors.push('Problem must be at least 100 characters');
  }
  if (!data.solution || data.solution.length < 200) {
    errors.push('Solution must be at least 200 characters');
  }
  if (!data.category) {
    errors.push('Category is required');
  }
  if (!data.skills || data.skills.length === 0) {
    errors.push('At least one skill/tool is required');
  }
  
  return errors;
}

// Submit to API
async function submit(data) {
  return new Promise((resolve, reject) => {
    const payload = JSON.stringify(data);
    
    const options = {
      hostname: API_URL,
      port: 443,
      path: API_PATH,
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Content-Length': Buffer.byteLength(payload)
      }
    };
    
    const req = https.request(options, (res) => {
      let body = '';
      
      res.on('data', (chunk) => {
        body += chunk;
      });
      
      res.on('end', () => {
        try {
          const response = JSON.parse(body);
          
          if (res.statusCode === 200 || res.statusCode === 201) {
            resolve(response);
          } else {
            reject(new Error(response.error || `HTTP ${res.statusCode}: ${body}`));
          }
        } catch (err) {
          reject(new Error(`Failed to parse response: ${body}`));
        }
      });
    });
    
    req.on('error', (err) => {
      reject(new Error(`Request failed: ${err.message}`));
    });
    
    req.write(payload);
    req.end();
  });
}

// Main
async function main() {
  const args = parseArgs();
  
  // Build submission data
  const data = {
    title: args.title,
    hook: args.hook,
    problem: args.problem,
    solution: args.solution,
    category: args.category,
    skills: args.skills ? args.skills.split(',').map(s => s.trim()) : [],
    requirements: args.requirements || undefined,
  };
  
  // Add author info if not anonymous
  if (!args.anonymous) {
    if (!args.author_username || !args.author_handle || !args.author_platform) {
      console.error('❌ Missing author info. Either provide --author-username, --author-handle, --author-platform, or use --anonymous');
      process.exit(1);
    }
    
    data.author = {
      username: args.author_username,
      handle: args.author_handle,
      platform: args.author_platform,
      link: args.author_link || undefined,
    };
  } else {
    // Anonymous submission
    data.author = {
      username: 'anonymous',
      handle: 'Anonymous',
      platform: 'anonymous',
    };
  }
  
  // Validate
  const errors = validate(data);
  if (errors.length > 0) {
    console.error('❌ Validation failed:');
    errors.forEach(err => console.error(`  - ${err}`));
    process.exit(1);
  }
  
  // Submit
  try {
    console.error('📤 Submitting use case...');
    const result = await submit(data);
    
    // Output result as JSON for the assistant to parse
    console.log(JSON.stringify(result, null, 2));
    
  } catch (err) {
    console.error('❌ Submission failed:', err.message);
    
    // Check for specific error codes
    if (err.message.includes('429')) {
      console.error('⏰ Rate limit reached (10 submissions per day)');
      console.error('Try again tomorrow!');
    } else if (err.message.includes('400')) {
      console.error('📝 Validation error - check your inputs');
    }
    
    process.exit(1);
  }
}

main();
