#!/usr/bin/env node

/**
 * Normalize tool/skill names for consistency
 * 
 * Usage:
 *   node normalize-tools.js "github,stripe api,resend email,react js"
 * 
 * Output:
 *   GitHub,Stripe,Resend,React
 */

// Tool normalization map
const TOOL_MAP = {
  // AI & ML
  'openai': 'OpenAI',
  'anthropic': 'Anthropic',
  'claude': 'Claude',
  'gpt': 'OpenAI',
  'chatgpt': 'OpenAI',
  'midjourney': 'Midjourney',
  'stable diffusion': 'Stable Diffusion',
  
  // Development
  'github': 'GitHub',
  'gitlab': 'GitLab',
  'bitbucket': 'Bitbucket',
  'git': 'Git',
  'vscode': 'VS Code',
  'visual studio code': 'VS Code',
  'docker': 'Docker',
  'kubernetes': 'Kubernetes',
  'k8s': 'Kubernetes',
  'terraform': 'Terraform',
  'ansible': 'Ansible',
  
  // Languages & Frameworks
  'javascript': 'JavaScript',
  'js': 'JavaScript',
  'typescript': 'TypeScript',
  'ts': 'TypeScript',
  'python': 'Python',
  'node': 'Node.js',
  'nodejs': 'Node.js',
  'node.js': 'Node.js',
  'react': 'React',
  'reactjs': 'React',
  'react.js': 'React',
  'vue': 'Vue',
  'vuejs': 'Vue',
  'vue.js': 'Vue',
  'angular': 'Angular',
  'next': 'Next.js',
  'nextjs': 'Next.js',
  'next.js': 'Next.js',
  'svelte': 'Svelte',
  'express': 'Express',
  'fastapi': 'FastAPI',
  'django': 'Django',
  'flask': 'Flask',
  
  // Databases
  'postgres': 'PostgreSQL',
  'postgresql': 'PostgreSQL',
  'mysql': 'MySQL',
  'mongodb': 'MongoDB',
  'mongo': 'MongoDB',
  'redis': 'Redis',
  'sqlite': 'SQLite',
  'dynamodb': 'DynamoDB',
  'firestore': 'Firestore',
  'supabase': 'Supabase',
  'planetscale': 'PlanetScale',
  
  // Cloud & Infrastructure
  'aws': 'AWS',
  'amazon web services': 'AWS',
  'gcp': 'Google Cloud',
  'google cloud': 'Google Cloud',
  'azure': 'Azure',
  'vercel': 'Vercel',
  'netlify': 'Netlify',
  'heroku': 'Heroku',
  'cloudflare': 'Cloudflare',
  'digitalocean': 'DigitalOcean',
  'railway': 'Railway',
  'fly': 'Fly.io',
  'fly.io': 'Fly.io',
  
  // APIs & Services
  'stripe': 'Stripe',
  'stripe api': 'Stripe',
  'paypal': 'PayPal',
  'twilio': 'Twilio',
  'sendgrid': 'SendGrid',
  'resend': 'Resend',
  'resend email': 'Resend',
  'mailgun': 'Mailgun',
  'postmark': 'Postmark',
  'slack': 'Slack',
  'slack api': 'Slack',
  'discord': 'Discord',
  'telegram': 'Telegram',
  'whatsapp': 'WhatsApp',
  'twitter': 'Twitter',
  'x': 'Twitter',
  
  // Backend & APIs
  'graphql': 'GraphQL',
  'rest': 'REST',
  'rest api': 'REST',
  'grpc': 'gRPC',
  'websocket': 'WebSocket',
  'webhooks': 'Webhooks',
  
  // Tools & Platforms
  'figma': 'Figma',
  'notion': 'Notion',
  'airtable': 'Airtable',
  'zapier': 'Zapier',
  'make': 'Make',
  'n8n': 'n8n',
  
  // Analytics & Monitoring
  'google analytics': 'Google Analytics',
  'posthog': 'PostHog',
  'mixpanel': 'Mixpanel',
  'amplitude': 'Amplitude',
  'sentry': 'Sentry',
  'datadog': 'Datadog',
  'newrelic': 'New Relic',
  'new relic': 'New Relic',
  
  // CMS & Content
  'wordpress': 'WordPress',
  'contentful': 'Contentful',
  'sanity': 'Sanity',
  'strapi': 'Strapi',
  
  // Testing
  'jest': 'Jest',
  'cypress': 'Cypress',
  'playwright': 'Playwright',
  'selenium': 'Selenium',
  'vitest': 'Vitest',
  
  // Build & Tooling
  'webpack': 'Webpack',
  'vite': 'Vite',
  'rollup': 'Rollup',
  'babel': 'Babel',
  'eslint': 'ESLint',
  'prettier': 'Prettier',
  
  // Home Automation
  'home assistant': 'Home Assistant',
  'homekit': 'HomeKit',
  'alexa': 'Alexa',
  'google home': 'Google Home',
  'smartthings': 'SmartThings',
  'philips hue': 'Philips Hue',
  'nest': 'Nest',
  'ring': 'Ring',
};

/**
 * Normalize a single tool name
 */
function normalizeTool(tool) {
  // Clean up
  const cleaned = tool
    .toLowerCase()
    .trim()
    .replace(/[^\w\s.-]/g, '') // Remove special chars except spaces, dots, hyphens
    .replace(/\s+/g, ' '); // Normalize whitespace
  
  // Check map
  if (TOOL_MAP[cleaned]) {
    return TOOL_MAP[cleaned];
  }
  
  // Title case fallback
  return tool
    .trim()
    .split(' ')
    .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
    .join(' ');
}

/**
 * Normalize a list of tools
 */
function normalizeTools(input) {
  // Split by common delimiters
  const tools = input
    .split(/[,;|]/)
    .map(t => t.trim())
    .filter(t => t.length > 0);
  
  // Normalize each
  const normalized = tools.map(normalizeTool);
  
  // Remove duplicates
  return [...new Set(normalized)];
}

// Main
function main() {
  const input = process.argv[2];
  
  if (!input) {
    console.error('Usage: node normalize-tools.js "github,stripe,resend"');
    process.exit(1);
  }
  
  const normalized = normalizeTools(input);
  
  // Output as comma-separated for easy parsing
  console.log(normalized.join(','));
}

if (require.main === module) {
  main();
}

module.exports = { normalizeTool, normalizeTools };
