#!/usr/bin/env node

/**
 * Retrieve OAuth credential from Convex after user completes authentication
 * 
 * Usage:
 *   node get-credential.js --token abc123def456
 */

const https = require('https');
const fs = require('fs');
const path = require('path');

// Load config
const configPath = path.join(__dirname, 'config.json');
const config = fs.existsSync(configPath) 
  ? JSON.parse(fs.readFileSync(configPath, 'utf8'))
  : {};

const CONVEX_URL = process.env.CONVEX_URL || config.convexUrl || 'benevolent-tortoise-657.convex.cloud';

// Parse command line arguments
function parseArgs() {
  const args = process.argv.slice(2);
  const parsed = {};
  
  for (let i = 0; i < args.length; i += 2) {
    const key = args[i].replace(/^--/, '').replace(/-/g, '_');
    const value = args[i + 1];
    parsed[key] = value;
  }
  
  return parsed;
}

// Query Convex
async function queryConvex(functionName, args) {
  return new Promise((resolve, reject) => {
    const payload = JSON.stringify({
      path: functionName,
      args: [args],
      format: 'json',
    });
    
    const options = {
      hostname: CONVEX_URL,
      port: 443,
      path: '/api/query',
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Content-Length': Buffer.byteLength(payload),
      },
    };
    
    const req = https.request(options, (res) => {
      let body = '';
      
      res.on('data', (chunk) => {
        body += chunk;
      });
      
      res.on('end', () => {
        try {
          const response = JSON.parse(body);
          
          if (res.statusCode === 200) {
            resolve(response);
          } else {
            reject(new Error(response.error || `HTTP ${res.statusCode}: ${body}`));
          }
        } catch (err) {
          reject(new Error(`Failed to parse response: ${body}`));
        }
      });
    });
    
    req.on('error', (err) => {
      reject(new Error(`Request failed: ${err.message}`));
    });
    
    req.write(payload);
    req.end();
  });
}

// Main
async function main() {
  const args = parseArgs();
  
  if (!args.token) {
    console.error('❌ Missing --token argument');
    console.error('Usage: node get-credential.js --token abc123');
    process.exit(1);
  }
  
  try {
    console.error('🔍 Retrieving OAuth credential...');
    
    const result = await queryConvex('oauth:getToken', { token: args.token });
    
    if (!result.value) {
      console.error('❌ Token not found or expired');
      process.exit(1);
    }
    
    if (!result.value.credential) {
      console.error('⏳ Authentication not yet completed');
      console.error('Make sure the user has clicked the OAuth link and authorized the app.');
      process.exit(1);
    }
    
    // Output credential as JSON
    console.log(JSON.stringify(result.value.credential, null, 2));
    
  } catch (err) {
    console.error('❌ Failed to retrieve credential:', err.message);
    process.exit(1);
  }
}

main();
