#!/bin/bash
# verify-report.sh - Verify SEV-SNP attestation report signature
#
# Usage: ./verify-report.sh <report_file> <certs_dir>
#
# Verifies:
#   1. Report signature using VCEK public key (ECDSA P-384)
#   2. Report structure integrity

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

usage() {
    echo "Usage: $0 <report_file> <certs_dir>"
    echo ""
    echo "Arguments:"
    echo "  report_file - Path to attestation report.bin"
    echo "  certs_dir   - Directory containing vcek.pem"
    exit 1
}

if [[ $# -lt 2 ]]; then
    usage
fi

REPORT_FILE="$1"
CERTS_DIR="$2"

if [[ ! -f "$REPORT_FILE" ]]; then
    echo -e "${RED}Error: Report file not found: $REPORT_FILE${NC}"
    exit 1
fi

if [[ ! -f "$CERTS_DIR/vcek.pem" ]]; then
    echo -e "${RED}Error: VCEK certificate not found: $CERTS_DIR/vcek.pem${NC}"
    exit 1
fi

echo "=== SEV-SNP Report Signature Verification ==="
echo ""
echo "Report file: $REPORT_FILE"
echo "VCEK cert:   $CERTS_DIR/vcek.pem"
echo ""

# Check for snpguest (preferred method)
if command -v snpguest &>/dev/null; then
    echo "Verifying report using snpguest..."
    echo ""

    if snpguest verify attestation "$CERTS_DIR" "$REPORT_FILE" 2>&1; then
        echo ""
        echo -e "${GREEN}Report signature verification PASSED${NC}"

        # Display report details
        echo ""
        echo "=== Attestation Report Details ==="
        snpguest display report "$REPORT_FILE" 2>/dev/null || true

        exit 0
    else
        echo ""
        echo -e "${RED}Report signature verification FAILED${NC}"
        exit 1
    fi
fi

# Fallback: Manual verification using OpenSSL
echo "snpguest not found, attempting manual verification with OpenSSL..."
echo ""

# Check dependencies for manual verification
for cmd in openssl python3 xxd; do
    if ! command -v $cmd &>/dev/null; then
        echo -e "${RED}Error: $cmd not found (required for manual verification)${NC}"
        exit 1
    fi
done

# SEV-SNP report structure:
# - Total size: 0x4A0 (1184) bytes
# - Signature at offset 0x2A0 (672), 144 bytes (R: 72 bytes, S: 72 bytes, but padded)
# - Signed data: bytes 0x00 to 0x29F (672 bytes)

REPORT_SIZE=$(stat -c %s "$REPORT_FILE")
EXPECTED_SIZE=1184

if [[ $REPORT_SIZE -lt $EXPECTED_SIZE ]]; then
    echo -e "${RED}Error: Report file too small (${REPORT_SIZE} bytes, expected ${EXPECTED_SIZE})${NC}"
    exit 1
fi

WORK_DIR=$(mktemp -d)
trap 'rm -rf "$WORK_DIR"' EXIT

# Extract signed data (first 672 bytes)
dd if="$REPORT_FILE" of="$WORK_DIR/signed_data.bin" bs=1 count=672 2>/dev/null

# Extract signature (144 bytes at offset 672)
# The signature is in AMD's format: R (72 bytes, little-endian) || S (72 bytes, little-endian)
dd if="$REPORT_FILE" of="$WORK_DIR/sig_raw.bin" bs=1 skip=672 count=144 2>/dev/null

# Convert AMD signature format to DER format for OpenSSL
# AMD uses little-endian 576-bit integers padded to 72 bytes
# OpenSSL expects DER-encoded ECDSA signature

# Extract R and S (each 48 bytes significant for P-384, but stored as 72 bytes LE)
# We need to reverse byte order and take last 48 bytes
dd if="$WORK_DIR/sig_raw.bin" of="$WORK_DIR/r_le.bin" bs=1 count=72 2>/dev/null
dd if="$WORK_DIR/sig_raw.bin" of="$WORK_DIR/s_le.bin" bs=1 skip=72 count=72 2>/dev/null

# Reverse byte order (LE to BE) and extract 48 bytes for P-384
python3 - "$WORK_DIR" <<'PYEOF'
import sys
work_dir = sys.argv[1]

def read_le_to_be(filepath, out_path):
    with open(filepath, 'rb') as f:
        data = f.read()
    # Reverse for big-endian, strip leading zeros but keep at least 48 bytes
    be_data = data[::-1]
    # For P-384, we need exactly 48 bytes
    # Find first non-zero byte
    first_nonzero = 0
    for i, b in enumerate(be_data):
        if b != 0:
            first_nonzero = i
            break
    # Take from first non-zero, pad to 48 if needed
    trimmed = be_data[first_nonzero:]
    if len(trimmed) < 48:
        trimmed = b'\x00' * (48 - len(trimmed)) + trimmed
    elif len(trimmed) > 48:
        trimmed = trimmed[-48:]
    with open(out_path, 'wb') as f:
        f.write(trimmed)

read_le_to_be(f'{work_dir}/r_le.bin', f'{work_dir}/r_be.bin')
read_le_to_be(f'{work_dir}/s_le.bin', f'{work_dir}/s_be.bin')

# Create DER-encoded signature
# SEQUENCE { INTEGER r, INTEGER s }
def to_der_int(data):
    # If high bit set, prepend 0x00
    if data[0] & 0x80:
        data = b'\x00' + data
    return bytes([0x02, len(data)]) + data

with open(f'{work_dir}/r_be.bin', 'rb') as f:
    r = f.read()
with open(f'{work_dir}/s_be.bin', 'rb') as f:
    s = f.read()

r_der = to_der_int(r)
s_der = to_der_int(s)
seq_content = r_der + s_der
sig_der = bytes([0x30, len(seq_content)]) + seq_content

with open(f'{work_dir}/signature.der', 'wb') as f:
    f.write(sig_der)

print(f"DER signature created: {len(sig_der)} bytes")
PYEOF

# Extract public key from VCEK
openssl x509 -in "$CERTS_DIR/vcek.pem" -pubkey -noout > "$WORK_DIR/vcek_pub.pem"

# Verify signature
echo "Verifying ECDSA P-384 signature..."
if openssl dgst -sha384 -verify "$WORK_DIR/vcek_pub.pem" -signature "$WORK_DIR/signature.der" "$WORK_DIR/signed_data.bin" 2>/dev/null; then
    echo ""
    echo -e "${GREEN}Report signature verification PASSED${NC}"

    # Display basic report info
    echo ""
    echo "=== Report Summary ==="
    echo "Report version: $(xxd -p -s 0 -l 4 "$REPORT_FILE" | fold -w2 | tac | tr -d '\n')"

    # MEASUREMENT at offset 0x90, 48 bytes
    echo "Measurement:    $(xxd -p -s 144 -l 48 "$REPORT_FILE" | tr -d '\n')"

    # REPORT_DATA at offset 0x50, 64 bytes
    echo "Report data:    $(xxd -p -s 80 -l 32 "$REPORT_FILE" | tr -d '\n')..."

    exit 0
else
    echo ""
    echo -e "${RED}Report signature verification FAILED${NC}"
    echo ""
    echo "Possible causes:"
    echo "  - Report was tampered with"
    echo "  - VCEK certificate doesn't match the chip that generated the report"
    echo "  - Report format is incompatible"
    exit 1
fi
