#!/bin/bash
# verify-chain.sh - Verify AMD certificate chain
#
# Usage: ./verify-chain.sh <certs_dir>
#
# Verifies:
#   1. ARK is self-signed (AMD Root Key)
#   2. ASK is signed by ARK (AMD SEV Key)
#   3. VCEK is signed by ASK (Versioned Chip Endorsement Key)

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

usage() {
    echo "Usage: $0 <certs_dir>"
    echo ""
    echo "Arguments:"
    echo "  certs_dir - Directory containing ark.pem, ask.pem, and vcek.pem"
    exit 1
}

if [[ $# -lt 1 ]]; then
    usage
fi

CERTS_DIR="$1"

# Verify required files exist
for cert in ark.pem ask.pem vcek.pem; do
    if [[ ! -f "$CERTS_DIR/$cert" ]]; then
        echo -e "${RED}Error: Missing certificate: $CERTS_DIR/$cert${NC}"
        exit 1
    fi
done

echo "=== AMD Certificate Chain Verification ==="
echo ""
echo "Certificate directory: $CERTS_DIR"
echo ""

CHAIN_VALID=true

# Step 1: Verify ARK is self-signed
echo "Step 1: Verifying ARK is self-signed..."
if openssl verify -CAfile "$CERTS_DIR/ark.pem" "$CERTS_DIR/ark.pem" 2>/dev/null | grep -q "OK"; then
    echo -e "  ${GREEN}✓${NC} ARK is self-signed (AMD Root Key)"

    # Show ARK details
    ARK_SUBJECT=$(openssl x509 -in "$CERTS_DIR/ark.pem" -noout -subject 2>/dev/null | sed 's/subject=//')
    ARK_ISSUER=$(openssl x509 -in "$CERTS_DIR/ark.pem" -noout -issuer 2>/dev/null | sed 's/issuer=//')
    echo "     Subject: $ARK_SUBJECT"
    echo "     Issuer:  $ARK_ISSUER"
else
    echo -e "  ${RED}✗${NC} ARK self-signature verification FAILED"
    CHAIN_VALID=false
fi

echo ""

# Step 2: Verify ASK is signed by ARK
echo "Step 2: Verifying ASK is signed by ARK..."
if openssl verify -CAfile "$CERTS_DIR/ark.pem" "$CERTS_DIR/ask.pem" 2>/dev/null | grep -q "OK"; then
    echo -e "  ${GREEN}✓${NC} ASK is signed by ARK (AMD SEV Key)"

    # Show ASK details
    ASK_SUBJECT=$(openssl x509 -in "$CERTS_DIR/ask.pem" -noout -subject 2>/dev/null | sed 's/subject=//')
    echo "     Subject: $ASK_SUBJECT"
else
    echo -e "  ${RED}✗${NC} ASK signature verification FAILED"
    echo "     ASK should be signed by ARK"
    CHAIN_VALID=false
fi

echo ""

# Step 3: Verify VCEK is signed by ASK
echo "Step 3: Verifying VCEK is signed by ASK..."

# Create certificate bundle (ARK + ASK) for VCEK verification
cat "$CERTS_DIR/ark.pem" "$CERTS_DIR/ask.pem" > "$CERTS_DIR/ca_bundle.pem"

if openssl verify -CAfile "$CERTS_DIR/ca_bundle.pem" "$CERTS_DIR/vcek.pem" 2>/dev/null | grep -q "OK"; then
    echo -e "  ${GREEN}✓${NC} VCEK is signed by ASK (Versioned Chip Endorsement Key)"

    # Show VCEK details
    VCEK_SUBJECT=$(openssl x509 -in "$CERTS_DIR/vcek.pem" -noout -subject 2>/dev/null | sed 's/subject=//')
    echo "     Subject: $VCEK_SUBJECT"

    # Extract and display TCB info from VCEK extensions if available
    echo ""
    echo "  VCEK Certificate Extensions:"
    # The VCEK has custom extensions with TCB info
    openssl x509 -in "$CERTS_DIR/vcek.pem" -noout -text 2>/dev/null | grep -A1 "hwID\|CHIP_ID\|1.3.6.1.4.1.3704" | head -10 || true
else
    echo -e "  ${RED}✗${NC} VCEK signature verification FAILED"
    echo "     VCEK should be signed by ASK"
    CHAIN_VALID=false
fi

# Cleanup
rm -f "$CERTS_DIR/ca_bundle.pem"

echo ""
echo "=== Chain Verification Summary ==="

if $CHAIN_VALID; then
    echo -e "${GREEN}Certificate chain is VALID${NC}"
    echo ""
    echo "Chain: ARK (AMD Root) → ASK (AMD SEV) → VCEK (Chip-specific)"
    echo ""
    echo "Next step: Verify attestation report signature"
    echo "  ./scripts/verify-report.sh <report.bin> $CERTS_DIR"
    exit 0
else
    echo -e "${RED}Certificate chain verification FAILED${NC}"
    echo ""
    echo "Possible causes:"
    echo "  - Certificates are corrupted or tampered"
    echo "  - Certificates are from different chip/platform"
    echo "  - Certificate files are in wrong format"
    exit 1
fi
