#!/bin/bash
# generate-report.sh - Generate an AMD SEV-SNP attestation report
#
# Usage: ./generate-report.sh <output_dir> [report_data_hex]
#
# Arguments:
#   output_dir      - Directory to store report.bin and related files
#   report_data_hex - Optional 64-byte hex string for REPORT_DATA field
#                     If not provided, a random nonce is generated
#
# Outputs:
#   <output_dir>/report.bin   - Raw attestation report
#   <output_dir>/nonce.hex    - The nonce/report_data used

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

usage() {
    echo "Usage: $0 <output_dir> [report_data_hex]"
    echo ""
    echo "Arguments:"
    echo "  output_dir      - Directory to store report.bin and related files"
    echo "  report_data_hex - Optional 64-byte hex string for REPORT_DATA field"
    echo "                    If not provided, a random nonce is generated"
    exit 1
}

if [[ $# -lt 1 ]]; then
    usage
fi

OUTPUT_DIR="$1"
REPORT_DATA_HEX="${2:-}"

# Check dependencies
for cmd in snpguest openssl xxd; do
    if ! command -v $cmd &>/dev/null; then
        echo -e "${RED}Error: $cmd not found${NC}"
        if [[ "$cmd" == "snpguest" ]]; then
            echo "Install with: cargo install snpguest"
        fi
        exit 1
    fi
done

# Check for /dev/sev-guest
if [[ ! -c /dev/sev-guest ]]; then
    echo -e "${RED}Error: /dev/sev-guest not found${NC}"
    echo "SEV-SNP is not available on this system."
    echo "Run ./scripts/detect-sev-snp.sh for more details."
    exit 1
fi

# Create output directory
mkdir -p "$OUTPUT_DIR"

echo "=== Generating SEV-SNP Attestation Report ==="
echo ""

# Generate or use provided report data (nonce)
if [[ -z "$REPORT_DATA_HEX" ]]; then
    echo "Generating random nonce (64 bytes)..."
    REPORT_DATA_HEX=$(openssl rand -hex 64)
else
    # Validate hex string length (64 bytes = 128 hex chars)
    if [[ ${#REPORT_DATA_HEX} -ne 128 ]]; then
        echo -e "${RED}Error: report_data_hex must be exactly 128 hex characters (64 bytes)${NC}"
        exit 1
    fi
fi

echo "Nonce: ${REPORT_DATA_HEX:0:32}..."
echo "$REPORT_DATA_HEX" > "$OUTPUT_DIR/nonce.hex"

# Generate attestation report
echo ""
echo "Requesting attestation report from AMD PSP..."

# snpguest report requires the data as a file
echo -n "$REPORT_DATA_HEX" | xxd -r -p > "$OUTPUT_DIR/request_data.bin"

if snpguest report "$OUTPUT_DIR/report.bin" "$OUTPUT_DIR/request_data.bin" 2>&1; then
    echo -e "${GREEN}Report generated successfully${NC}"
else
    echo -e "${RED}Failed to generate attestation report${NC}"
    echo "Check permissions: /dev/sev-guest requires root or sev group"
    exit 1
fi

# Display report information
echo ""
echo "=== Report Summary ==="
if snpguest display report "$OUTPUT_DIR/report.bin" 2>/dev/null; then
    :
else
    echo -e "${YELLOW}Warning: Could not display report details${NC}"
    echo "The report was generated but snpguest display failed."
fi

echo ""
echo "=== Output Files ==="
echo "  Report:     $OUTPUT_DIR/report.bin"
echo "  Nonce:      $OUTPUT_DIR/nonce.hex"
echo ""
echo "Next step: Fetch AMD certificates"
echo "  ./scripts/fetch-certificates.sh $OUTPUT_DIR/report.bin $OUTPUT_DIR"
