#!/bin/bash
# full-attestation.sh - Complete SEV-SNP attestation workflow
#
# Usage: ./full-attestation.sh [output_dir]
#
# Runs the complete 6-step attestation workflow:
#   1. Detection - Check SEV-SNP availability
#   2. Report generation - Request attestation from AMD PSP
#   3. Display report info
#   4. Fetch certificates - Get ARK, ASK, VCEK from AMD KDS
#   5. Verify chain - Validate certificate chain
#   6. Verify report - Validate report signature
#
# Exit codes:
#   0 - Attestation PASSED
#   1 - Attestation FAILED

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Get script directory for relative paths
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Default output directory
OUTPUT_DIR="${1:-./attestation-$(date +%Y%m%d-%H%M%S)}"

echo "╔══════════════════════════════════════════════════════════════════╗"
echo "║           AMD SEV-SNP Remote Attestation Workflow                 ║"
echo "╚══════════════════════════════════════════════════════════════════╝"
echo ""
echo "Output directory: $OUTPUT_DIR"
echo ""

# Create output directory
mkdir -p "$OUTPUT_DIR"

# Track overall status
ATTESTATION_PASSED=true

# Step 1: Detection
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 1/6: SEV-SNP Detection                                    │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if "$SCRIPT_DIR/detect-sev-snp.sh"; then
    echo ""
    echo -e "${GREEN}✓ Step 1 PASSED: SEV-SNP is available${NC}"
else
    echo ""
    echo -e "${RED}✗ Step 1 FAILED: SEV-SNP is not available${NC}"
    echo ""
    echo "╔══════════════════════════════════════════════════════════════════╗"
    echo -e "║                    ${RED}ATTESTATION FAILED${NC}                            ║"
    echo "╚══════════════════════════════════════════════════════════════════╝"
    echo ""
    echo "SEV-SNP is not available on this system. Attestation cannot proceed."
    exit 1
fi

echo ""
echo ""

# Step 2: Generate Report
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 2/6: Generate Attestation Report                          │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if "$SCRIPT_DIR/generate-report.sh" "$OUTPUT_DIR"; then
    echo ""
    echo -e "${GREEN}✓ Step 2 PASSED: Attestation report generated${NC}"
else
    echo ""
    echo -e "${RED}✗ Step 2 FAILED: Could not generate attestation report${NC}"
    ATTESTATION_PASSED=false
fi

echo ""
echo ""

# Step 3: Display Report Info (informational, doesn't affect pass/fail)
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 3/6: Display Report Information                           │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if [[ -f "$OUTPUT_DIR/report.bin" ]]; then
    if command -v snpguest &>/dev/null; then
        snpguest display report "$OUTPUT_DIR/report.bin" 2>/dev/null || echo "Could not display report details"
    else
        echo "Report generated at: $OUTPUT_DIR/report.bin"
        echo "Install snpguest for detailed report display"
    fi
    echo ""
    echo -e "${GREEN}✓ Step 3 PASSED: Report information displayed${NC}"
else
    echo -e "${YELLOW}⚠ Step 3 SKIPPED: No report file found${NC}"
fi

echo ""
echo ""

# Step 4: Fetch Certificates
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 4/6: Fetch AMD Certificates                               │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if [[ -f "$OUTPUT_DIR/report.bin" ]]; then
    if "$SCRIPT_DIR/fetch-certificates.sh" "$OUTPUT_DIR/report.bin" "$OUTPUT_DIR"; then
        echo ""
        echo -e "${GREEN}✓ Step 4 PASSED: Certificates fetched from AMD KDS${NC}"
    else
        echo ""
        echo -e "${RED}✗ Step 4 FAILED: Could not fetch certificates${NC}"
        ATTESTATION_PASSED=false
    fi
else
    echo -e "${YELLOW}⚠ Step 4 SKIPPED: No report file available${NC}"
    ATTESTATION_PASSED=false
fi

echo ""
echo ""

# Step 5: Verify Certificate Chain
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 5/6: Verify Certificate Chain                             │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if [[ -d "$OUTPUT_DIR/certs" ]]; then
    if "$SCRIPT_DIR/verify-chain.sh" "$OUTPUT_DIR/certs"; then
        echo ""
        echo -e "${GREEN}✓ Step 5 PASSED: Certificate chain is valid${NC}"
    else
        echo ""
        echo -e "${RED}✗ Step 5 FAILED: Certificate chain verification failed${NC}"
        ATTESTATION_PASSED=false
    fi
else
    echo -e "${YELLOW}⚠ Step 5 SKIPPED: No certificates directory${NC}"
    ATTESTATION_PASSED=false
fi

echo ""
echo ""

# Step 6: Verify Report Signature
echo "┌────────────────────────────────────────────────────────────────┐"
echo "│ Step 6/6: Verify Report Signature                              │"
echo "└────────────────────────────────────────────────────────────────┘"
echo ""

if [[ -f "$OUTPUT_DIR/report.bin" ]] && [[ -d "$OUTPUT_DIR/certs" ]]; then
    if "$SCRIPT_DIR/verify-report.sh" "$OUTPUT_DIR/report.bin" "$OUTPUT_DIR/certs"; then
        echo ""
        echo -e "${GREEN}✓ Step 6 PASSED: Report signature is valid${NC}"
    else
        echo ""
        echo -e "${RED}✗ Step 6 FAILED: Report signature verification failed${NC}"
        ATTESTATION_PASSED=false
    fi
else
    echo -e "${YELLOW}⚠ Step 6 SKIPPED: Missing report or certificates${NC}"
    ATTESTATION_PASSED=false
fi

echo ""
echo ""

# Final Summary
echo "╔══════════════════════════════════════════════════════════════════╗"
if $ATTESTATION_PASSED; then
    echo -e "║                    ${GREEN}ATTESTATION PASSED${NC}                            ║"
    echo "╚══════════════════════════════════════════════════════════════════╝"
    echo ""
    echo "The VM's identity has been cryptographically verified:"
    echo "  - Running on genuine AMD SEV-SNP hardware"
    echo "  - Certificate chain traces back to AMD root"
    echo "  - Attestation report signature is valid"
    echo ""
    echo "Output files saved to: $OUTPUT_DIR"
    echo "  - report.bin    : Attestation report"
    echo "  - nonce.hex     : Freshness nonce used"
    echo "  - certs/        : AMD certificate chain"
    exit 0
else
    echo -e "║                    ${RED}ATTESTATION FAILED${NC}                            ║"
    echo "╚══════════════════════════════════════════════════════════════════╝"
    echo ""
    echo "One or more verification steps failed. Review the output above"
    echo "for details on which step failed and why."
    echo ""
    echo "See references/error-codes.md for troubleshooting guidance."
    exit 1
fi
