#!/bin/bash
# fetch-certificates.sh - Fetch AMD certificates from KDS
#
# Usage: ./fetch-certificates.sh <report_file> <output_dir>
#
# Fetches:
#   - ARK (AMD Root Key) - Self-signed root certificate
#   - ASK (AMD SEV Key) - Intermediate certificate signed by ARK
#   - VCEK (Versioned Chip Endorsement Key) - Chip-specific cert signed by ASK
#
# The VCEK is chip-specific and requires data from the attestation report.

set -euo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

AMD_KDS_URL="https://kdsintf.amd.com"

usage() {
    echo "Usage: $0 <report_file> <output_dir>"
    echo ""
    echo "Arguments:"
    echo "  report_file - Path to the attestation report.bin"
    echo "  output_dir  - Directory to store certificates"
    exit 1
}

if [[ $# -lt 2 ]]; then
    usage
fi

REPORT_FILE="$1"
OUTPUT_DIR="$2"

if [[ ! -f "$REPORT_FILE" ]]; then
    echo -e "${RED}Error: Report file not found: $REPORT_FILE${NC}"
    exit 1
fi

# Check dependencies
for cmd in snpguest curl openssl xxd; do
    if ! command -v $cmd &>/dev/null; then
        echo -e "${RED}Error: $cmd not found${NC}"
        if [[ "$cmd" == "snpguest" ]]; then
            echo "Install with: cargo install snpguest"
        fi
        exit 1
    fi
done

mkdir -p "$OUTPUT_DIR/certs"

echo "=== Fetching AMD Certificates from KDS ==="
echo ""
echo "KDS URL: $AMD_KDS_URL"
echo ""

# Method 1: Try using snpguest fetch (preferred)
echo "Attempting certificate fetch via snpguest..."

if snpguest fetch ca pem milan "$OUTPUT_DIR/certs" 2>/dev/null; then
    echo -e "${GREEN}Fetched ARK and ASK certificates${NC}"
else
    echo -e "${YELLOW}snpguest fetch ca failed, trying manual fetch...${NC}"

    # Manual fetch of ARK and ASK
    echo "Fetching ARK (AMD Root Key)..."
    curl -sf "$AMD_KDS_URL/vcek/v1/Milan/cert_chain" -o "$OUTPUT_DIR/certs/cert_chain.pem" || {
        echo -e "${RED}Failed to fetch certificate chain${NC}"
        exit 1
    }

    # Split the chain into ARK and ASK
    # The chain typically has ASK first, then ARK
    csplit -sf "$OUTPUT_DIR/certs/cert_" -b "%d.pem" "$OUTPUT_DIR/certs/cert_chain.pem" '/-----BEGIN CERTIFICATE-----/' '{*}' 2>/dev/null || true

    # Rename appropriately (cert_1.pem is ASK, cert_2.pem is ARK)
    if [[ -f "$OUTPUT_DIR/certs/cert_1.pem" ]]; then
        mv "$OUTPUT_DIR/certs/cert_1.pem" "$OUTPUT_DIR/certs/ask.pem"
    fi
    if [[ -f "$OUTPUT_DIR/certs/cert_2.pem" ]]; then
        mv "$OUTPUT_DIR/certs/cert_2.pem" "$OUTPUT_DIR/certs/ark.pem"
    fi
    rm -f "$OUTPUT_DIR/certs/cert_0.pem" "$OUTPUT_DIR/certs/cert_"*.pem 2>/dev/null || true

    echo -e "${GREEN}Fetched ARK and ASK certificates${NC}"
fi

# Fetch VCEK using report data
echo ""
echo "Fetching VCEK (Versioned Chip Endorsement Key)..."

# snpguest can fetch VCEK directly using the report
if snpguest fetch vcek pem milan "$OUTPUT_DIR/certs" "$REPORT_FILE" 2>/dev/null; then
    echo -e "${GREEN}Fetched VCEK certificate${NC}"
else
    echo -e "${YELLOW}snpguest fetch vcek failed, trying manual method...${NC}"

    # Extract CHIP_ID and TCB from report for manual VCEK fetch
    # This requires parsing the binary report
    # Report structure: CHIP_ID at offset 0x1A0 (416), TCB_VERSION at offset 0x38 (56)

    CHIP_ID=$(xxd -p -s 416 -l 64 "$REPORT_FILE" | tr -d '\n')

    # TCB version components (each 1 byte at specific offsets in TCB)
    # TCB_VERSION is at offset 0x38 (56) in the report, 8 bytes total
    TCB_BYTES=$(xxd -p -s 56 -l 8 "$REPORT_FILE" | tr -d '\n')

    # Parse TCB components: boot_loader, tee, snp, microcode (all in positions within TCB)
    BOOT_LOADER=$(printf "%d" "0x${TCB_BYTES:0:2}")
    TEE=$(printf "%d" "0x${TCB_BYTES:2:2}")
    SNP=$(printf "%d" "0x${TCB_BYTES:12:2}")
    MICROCODE=$(printf "%d" "0x${TCB_BYTES:14:2}")

    VCEK_URL="$AMD_KDS_URL/vcek/v1/Milan/${CHIP_ID}?blSPL=${BOOT_LOADER}&teeSPL=${TEE}&snpSPL=${SNP}&ucodeSPL=${MICROCODE}"

    echo "Fetching VCEK from: ${VCEK_URL:0:80}..."

    curl -sf "$VCEK_URL" -o "$OUTPUT_DIR/certs/vcek.der" || {
        echo -e "${RED}Failed to fetch VCEK certificate${NC}"
        echo "URL: $VCEK_URL"
        exit 1
    }

    # Convert DER to PEM
    openssl x509 -inform der -in "$OUTPUT_DIR/certs/vcek.der" -out "$OUTPUT_DIR/certs/vcek.pem" 2>/dev/null || {
        echo -e "${RED}Failed to convert VCEK to PEM${NC}"
        exit 1
    }

    echo -e "${GREEN}Fetched VCEK certificate${NC}"
fi

# Verify we have all certificates
echo ""
echo "=== Certificate Files ==="

CERTS_OK=true
for cert in ark.pem ask.pem vcek.pem; do
    CERT_PATH="$OUTPUT_DIR/certs/$cert"
    if [[ -f "$CERT_PATH" ]]; then
        SUBJECT=$(openssl x509 -in "$CERT_PATH" -noout -subject 2>/dev/null | sed 's/subject=//')
        echo -e "  ${GREEN}✓${NC} $cert: $SUBJECT"
    else
        echo -e "  ${RED}✗${NC} $cert: NOT FOUND"
        CERTS_OK=false
    fi
done

if $CERTS_OK; then
    echo ""
    echo -e "${GREEN}All certificates fetched successfully${NC}"
    echo ""
    echo "Next step: Verify certificate chain"
    echo "  ./scripts/verify-chain.sh $OUTPUT_DIR/certs"
else
    echo ""
    echo -e "${RED}Some certificates are missing${NC}"
    exit 1
fi
